<?php

namespace wdigital\users\models;

use DateTime;
use wdigital\users\traits\ModuleTrait;
use Yii;
use yii\base\Exception;
use yii\base\InvalidConfigException;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQueryInterface;
use yii\db\ActiveRecord;
use yii\helpers\FileHelper;
use yii\web\UploadedFile;

/**
 * Class Profile
 * @package wdigital\users\models
 * @property string $user_id [integer]
 * @property string $title [varchar(255)]
 * @property string $image [varchar(255)]
 * @property int $created_at [timestamp(0)]
 * @property int $updated_at [timestamp(0)]
 *
 * @property-read User $user
 */
class Profile extends ActiveRecord
{
    use ModuleTrait;

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%user_profile}}';
    }

    /**
     * @inheritDoc
     */
    public function behaviors(): array
    {
        return [
            [
                'class' => TimestampBehavior::class,
                'value' => (new DateTime())->format('Y-m-d H:i:s'),
            ]
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            [['title'], 'string', 'max' => 255],
            [['image'], 'image', 'extensions' => ['png', 'jpg', 'jpeg'], 'minWidth' => 200, 'maxWidth' => 200, 'minHeight' => 200, 'maxHeight' => 200],
            [['user_id'], 'integer'],
            [['user_id'], 'exist', 'targetRelation' => 'user', 'targetAttribute' => 'id'],
        ];
    }

    /**
     * @return ActiveQueryInterface
     * @throws InvalidConfigException
     */
    public function getUser(): ActiveQueryInterface
    {
        return $this->hasOne(static::getModule()->modelMap['User'], ['id' => 'user_id']);
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'title' => Yii::t('user', 'Name Surname'),
            'image' => Yii::t('user', 'Profile image'),
        ];
    }

    /**
     * @inheritDoc
     */
    public function load($data, $formName = null): bool
    {
        $load = parent::load($data, $formName);
        if ($load) {
            $this->image = UploadedFile::getInstance($this, 'image');
        }
        return $load;
    }

    /**
     * @inheritDoc
     * @throws Exception
     */
    public function beforeSave($insert): bool
    {
        if (!$insert && $this->image instanceof UploadedFile && $this->getOldAttribute('image') !== null) {
            $filePath = Yii::getAlias("{$this->module->profileImageUploadFolder}/{$this->getOldAttribute('image')}");
            if (is_file($filePath)) {
                FileHelper::unlink($filePath);
            }
        }
        if ($this->image instanceof UploadedFile) {
            $fileName = Yii::$app->security->generateRandomString() . ".{$this->image->getExtension()}";
            $this->image->saveAs("{$this->module->profileImageUploadFolder}/{$fileName}");
            $this->image = $fileName;
        }
        return parent::beforeSave($insert);
    }
}
