<?php

namespace wdigital\users\models;

use DateTime;
use RuntimeException;
use wdigital\users\traits\ModuleTrait;
use Yii;
use yii\base\Exception;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\Url;

/**
 * Token Active Record model.
 *
 * @property integer $user_id
 * @property string  $code
 * @property integer $created_at
 * @property integer $type
 * @property string  $url
 * @property bool    $isExpired
 * @property User    $user
 *
 * @author Dmitry Erofeev <dmeroff@gmail.com>
 */
class Token extends ActiveRecord
{
    use ModuleTrait;

    const TYPE_CONFIRMATION      = 0;
    const TYPE_RECOVERY          = 1;

    /**
     * @return ActiveQuery
     */
    public function getUser()
    {
        return $this->hasOne($this->module->modelMap['User'], ['id' => 'user_id']);
    }

    /**
     * @return string
     */
    public function getUrl()
    {
        switch ($this->type) {
            case self::TYPE_CONFIRMATION:
                $route = '/user/security/confirm';
                break;
            case self::TYPE_RECOVERY:
                $route = '/user/security/reset';
                break;
            default:
                throw new RuntimeException();
        }

        return Url::to([$route, 'id' => $this->user_id, 'code' => $this->code], true);
    }

    /**
     * @return bool Whether token has expired.
     */
    public function getIsExpired(): bool
    {
        $threshold = DateTime::createFromFormat('Y-m-d H:i:s', $this->created_at);
        if (!$threshold instanceof DateTime) {
            throw new RuntimeException('Could not get token expiration datetime');
        }
        return $threshold->modify("+{$this->module->recoverWithin} seconds") < (new DateTime());
    }

    /**
     * @inheritdoc
     * @throws Exception
     */
    public function beforeSave($insert): bool
    {
        if ($insert) {
            static::deleteAll(['user_id' => $this->user_id, 'type' => $this->type]);
            $this->setAttribute('created_at', (new DateTime())->format('Y-m-d H:i:s'));
            $this->setAttribute('code', Yii::$app->security->generateRandomString());
        }

        return parent::beforeSave($insert);
    }

    /** @inheritdoc */
    public static function tableName()
    {
        return '{{%token}}';
    }

    /** @inheritdoc */
    public static function primaryKey()
    {
        return ['user_id', 'code', 'type'];
    }
}
