<?php

namespace wdigital\users\models;

use Exception;
use wdigital\users\validators\RbacValidator;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\helpers\ArrayHelper;
use yii\rbac\ManagerInterface;

/**
 * Class AssignmentForm
 * @package wdigital\users\models
 */
class AssignmentForm extends Model
{
    /**
     * @var array
     */
    public array $items = [];

    /**
     * @var int|null
     */
    public ?int $user_id = null;

    /**
     * @var boolean
     */
    public bool $updated = false;

    /**
     * @var ManagerInterface|null
     */
    protected ?ManagerInterface $manager = null;

    /**
     * @inheritdoc
     * @throws InvalidConfigException
     */
    public function init(): void
    {
        $this->manager = Yii::$app->authManager;
        if ($this->user_id === null) {
            throw new InvalidConfigException('user_id must be set');
        }

        $this->items = array_keys($this->manager->getRolesByUser($this->user_id));
        parent::init();
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'items' => Yii::t('user', 'Items'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            ['user_id', 'required'],
            [['items'], RbacValidator::class],
            ['user_id', 'integer']
        ];
    }

    /**
     * Updates auth assignments for user.
     * @return boolean
     * @throws Exception
     */
    public function updateAssignments(): bool
    {
        if (!$this->validate()) {
            return false;
        }

        if (!is_array($this->items)) {
            $this->items = [];
        }

        $assignedItems = $this->manager->getRolesByUser($this->user_id);
        $assignedItemsNames = array_keys($assignedItems);

        foreach (array_diff($assignedItemsNames, $this->items) as $item) {
            $this->manager->revoke($assignedItems[$item], $this->user_id);
        }

        foreach (array_diff($this->items, $assignedItemsNames) as $item) {
            $this->manager->assign($this->manager->getRole($item), $this->user_id);
        }

        $this->updated = true;

        return true;
    }

    /**
     * Returns all available auth items to be attached to user.
     * @return array
     */
    public function retrieveAvailableItems(): array
    {
        return ArrayHelper::map($this->manager->getRoles(), 'name', function ($item) {
            return empty($item->description)
                ? $item->name
                : $item->name . ' (' . $item->description . ')';
        });
    }

}
