<?php

namespace wdigital\users\controllers;

use wdigital\users\models\RbacItemSearch;
use Yii;
use yii\base\ExitException;
use yii\base\InvalidConfigException;
use yii\rbac\Permission;
use yii\rbac\Role;
use yii\web\Controller;
use yii\web\Response;

abstract class RbacItemControllerAbstract extends Controller
{
    /**
     * @param string $name
     * @return Role|Permission
     */
    abstract protected function getItem(string $name);

    /**
     * @var int|null
     */
    protected ?int $type = null;

    /**
     * @var string|null
     */
    protected ?string $modelClass = null;

    /**
     * @inheritDoc
     * @throws InvalidConfigException
     */
    public function init(): void
    {
        parent::init();
        if ($this->modelClass === null) {
            throw new InvalidConfigException('Model class should be set');
        }
        if ($this->type === null) {
            throw new InvalidConfigException('Auth item type should be set');
        }
    }

    /**
     * Lists all created items.
     * @return string
     * @throws InvalidConfigException
     */
    public function actionIndex(): string
    {
        $filterModel = new RbacItemSearch($this->type);
        return $this->render('index', [
            'filterModel' => $filterModel,
            'dataProvider' => $filterModel->search(Yii::$app->request->get()),
        ]);
    }

    /**
     * Shows create form.
     * @return string|Response
     * @throws InvalidConfigException
     */
    public function actionCreate()
    {
        $model = Yii::createObject([
            'class' => $this->modelClass,
            'manager' => Yii::$app->authManager,
        ]);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            return $this->redirect(['index']);
        }

        return $this->render('create', [
            'model' => $model,
            'rules' => [],
            'permissions' => array_column(Yii::$app->authManager->getPermissions(), 'name', 'name'),
        ]);
    }

    /**
     * Shows update form.
     * @param string $name
     * @return string|Response
     * @throws ExitException
     * @throws InvalidConfigException
     */
    public function actionUpdate(string $name)
    {
        $item = $this->getItem($name);
        $model = Yii::createObject([
            'class' => $this->modelClass,
            'manager' => Yii::$app->authManager,
            'item' => $item,
        ]);

        if ($model->load(Yii::$app->request->post()) && $model->save()) {
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
            'rules' => [],
            'permissions' => array_column(Yii::$app->authManager->getPermissions(), 'name', 'name'),
        ]);
    }

    /**
     * Deletes item.
     * @param string $name
     * @return Response
     */
    public function actionDelete(string $name): Response
    {
        $item = $this->getItem($name);
        Yii::$app->authManager->remove($item);
        return $this->redirect(['index']);
    }
}
