<?php

namespace wdigital\users\models;

use Exception;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\di\Instance;
use yii\rbac\DbManager;

/**
 * Rule model.
 */
class Rule extends Model
{
    public const SCENARIO_CREATE = 'create';
    public const SCENARIO_UPDATE = 'update';

    /**
     * @var string
     */
    public $name;

    /**
     * @var string
     */
    public $class;

    /**
     * @var string|DbManager The auth manager component ID.
     */
    public $authManager = 'authManager';

    /**
     * @var string
     */
    private $_oldName;

    /**
     * @param string $oldName
     */
    public function setOldName($oldName): void
    {
        $this->_oldName = $oldName;
    }

    /**
     * This method will set [[authManager]] to be the 'authManager' application component, if it is `null`.
     * @throws InvalidConfigException
     */
    public function init(): void
    {
        parent::init();

        $this->authManager = Instance::ensure($this->authManager, DbManager::class);
    }

    /**
     * @return array
     */
    public function scenarios(): array
    {
        return [
            self::SCENARIO_CREATE => ['name', 'class'],
            self::SCENARIO_UPDATE => ['name', 'class'],
        ];
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        return [
            [['name', 'class'], 'trim'],
            [['name', 'class'], 'required'],
            ['name', 'match', 'pattern' => '/^[\w][\w-.:]+[\w]$/'],
            ['name', function () {
                if ($this->name == $this->_oldName) {
                    return;
                }
                $rule = $this->authManager->getRule($this->name);

                if ($rule instanceof \yii\rbac\Rule) {
                    $this->addError('name', Yii::t('rbac', 'Name is already in use'));
                }
            }],
            ['class', function () {
                if (!class_exists($this->class)) {
                    $this->addError('class', Yii::t('rbac', 'Class "{0}" does not exist', $this->class));
                } else {
                    try {
                        $class = '\yii\rbac\Rule';
                        $rule  = Yii::createObject($this->class);

                        if (!($rule instanceof $class)) {
                            $this->addError('class', Yii::t('rbac', 'Rule class must extend "yii\rbac\Rule"'));
                        }
                    } catch (InvalidConfigException $e) {
                        $this->addError('class', Yii::t('rbac', 'Rule class can not be instantiated'));
                    }
                }
            }],
        ];
    }

    /**
     * Creates new auth rule.
     *
     * @return bool
     * @throws InvalidConfigException
     * @throws Exception
     */
    public function create(): bool
    {
        if ($this->scenario !== self::SCENARIO_CREATE) {
            return false;
        }

        if (!$this->validate()) {
            return false;
        }

        $rule = Yii::createObject([
            'class' => $this->class,
            'name'  => $this->name,
        ]);

        $this->authManager->add($rule);
        $this->authManager->invalidateCache();

        return true;
    }

    /**
     * Updates existing auth rule.
     *
     * @return bool
     * @throws InvalidConfigException
     * @throws Exception
     */
    public function update(): bool
    {
        if ($this->scenario !== self::SCENARIO_UPDATE) {
            return false;
        }

        if (!$this->validate()) {
            return false;
        }

        $rule = Yii::createObject([
            'class' => $this->class,
            'name'  => $this->name,
        ]);

        $this->authManager->update($this->_oldName, $rule);
        $this->authManager->invalidateCache();

        return true;
    }
}
