<?php

namespace wdigital\users\models;

use wdigital\users\validators\RbacValidator;
use Yii;
use yii\base\Model;
use yii\rbac\DbManager;
use yii\rbac\Permission;
use yii\rbac\Role;

/**
 * Class RbacPermissionForm
 * @package wdigital\users\models
 */
class RbacPermissionForm extends Model
{
    /**
     * @var Permission|null
     */
    public ?Permission $item = null;

    /**
     * @var string|null
     */
    public ?string $name = null;

    /**
     * @var string|null
     */
    public ?string $description = null;

    /**
     * @var mixed|null
     */
    public $children = null;

    /**
     * @var string|null
     */
    public ?string $rule_name = null;

    /**
     * @var DbManager
     */
    public ?DbManager $manager = null;

    /**
     * @inheritDoce
     */
    public function init(): void
    {
        parent::init();
        if ($this->item !== null) {
            $this->name = $this->item->name;
            $this->description = $this->item->description;
            $this->children = array_column($this->manager->getChildren($this->item->name), 'name');
            $this->rule_name = $this->item->ruleName;
        }
    }

    /**
     * @inheritDoc
     */
    public function rules(): array
    {
        return [
            [['name', 'description', 'rule_name'], 'trim'],
            [['name'], 'required'],
            [['name', 'rule_name'], 'string', 'max' => 64],
            [['description'], 'string'],
            [['children'], RbacValidator::class],
        ];
    }

    /**
     * @inheritDoc
     */
    public function attributeLabels(): array
    {
        return [
            'name' => Yii::t('user', 'Title'),
            'description' => Yii::t('user', 'Description'),
            'rule_name' => Yii::t('user', 'Rule name'),
            'children' => Yii::t('user', 'Children'),
        ];
    }

    /**
     * Saves item.
     *
     * @return bool
     * @throws \Exception
     */
    public function save(): bool
    {
        if (!$this->validate()) {
            return false;
        }
        if (!$this->item instanceof Permission) {
            $this->item = $this->manager->createPermission($this->name);
        } else {
            $oldName = $this->item->name;
        }
        $this->item->name = $this->name;
        $this->item->description = $this->description;
        $this->item->ruleName = $this->rule_name ?: null;

        if ($this->item->createdAt === null) {
            Yii::$app->session->setFlash('success', Yii::t('user', 'Permission has been created'));
            $this->manager->add($this->item);
        } else {
            Yii::$app->session->setFlash('success', Yii::t('user', 'Permission has been updated'));
            $this->manager->update($oldName, $this->item);
        }

        $this->updateChildren();

        $this->manager->invalidateCache();

        return true;
    }

    /**
     * Updated items children.
     */
    protected function updateChildren(): void
    {
        $oldChildren = array_keys($this->manager->getChildren($this->item->name));
        $oldChildrenNames = array_keys($oldChildren);

        if (is_array($this->children)) {
            foreach (array_diff($oldChildrenNames, $this->children) as $item) {
                $this->manager->removeChild($this->item, $oldChildren[$item]);
            }
            foreach (array_diff($this->children, $oldChildrenNames) as $item) {
                $this->manager->addChild($this->item, $this->manager->getPermission($item));
            }
        } else {
            $this->manager->removeChildren($this->item);
        }
    }
}
