<?php

namespace wdigital\users\models;

use Yii;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\data\ArrayDataProvider;
use yii\db\Query;
use yii\di\Instance;
use yii\helpers\ArrayHelper;
use yii\rbac\DbManager;

/**
 * Class RbacItemSearch
 * @package wdigital\users\models
 */
class RbacItemSearch extends Model
{
    /**
     * @var string|null
     */
    public ?string $name = null;

    /**
     * @var string|null
     */
    public ?string $description = null;

    /**
     * @var string|null
     */
    public ?string $rule_name = null;

    /**
     * @var DbManager
     */
    protected DbManager $manager;

    /**
     * @var int|null
     */
    protected ?int $type = null;

    /**
     * RbacItemSearch constructor.
     * @param int|null $type
     * @param array $config
     * @throws InvalidConfigException
     */
    public function __construct(?int $type = null, $config = [])
    {
        parent::__construct($config);
        $authManager = Yii::$app->getAuthManager();
        $this->manager = Instance::ensure($authManager, DbManager::class);
        $this->type = $type;
    }

    /**
     * @inheritdoc
     */
    public function scenarios(): array
    {
        return [
            'default' => ['name', 'description', 'rule_name'],
        ];
    }

    /**
     * @inheritDoc
     */
    public function attributeLabels(): array
    {
        return [
            'name' => Yii::t('user', 'Title'),
            'description' => Yii::t('user', 'Description'),
            'rule_name' => Yii::t('user', 'Rule name'),
        ];
    }

    /**
     * @param array $params
     * @return ArrayDataProvider
     * @throws InvalidConfigException
     */
    public function search(array $params = []): ArrayDataProvider
    {
        $dataProvider = Yii::createObject(ArrayDataProvider::class);

        $query = (new Query)->select(['name', 'description', 'rule_name'])
            ->andWhere(['type' => $this->type])
            ->from($this->manager->itemTable);

        if ($this->load($params) && $this->validate()) {
            $query->andFilterWhere(['like', 'name', $this->name])
                ->andFilterWhere(['like', 'description', $this->description])
                ->andFilterWhere(['like', 'rule_name', $this->rule_name]);
        }

        $dataProvider->allModels = $query->all($this->manager->db);

        return $dataProvider;
    }

    /**
     * Returns list of item names.
     *
     * @return array
     */
    public function getNameList(): array
    {
        $rows = (new Query)
            ->select(['name'])
            ->andWhere(['type' => $this->type])
            ->from($this->manager->itemTable)
            ->all();

        return ArrayHelper::map($rows, 'name', 'name');
    }

    /**
     * Returns list of rule names.
     *
     * @return array
     */
    public function getRuleList(): array
    {
        $rows = (new Query())
            ->select(['name'])
            ->from($this->manager->ruleTable)
            ->all();

        return ArrayHelper::map($rows, 'name', 'name');
    }
}
