<?php

namespace wdigital\users\models;

use DateTime;
use Exception;
use RuntimeException;
use wdigital\users\Finder;
use wdigital\users\helpers\Password;
use wdigital\users\Mailer;
use wdigital\users\Module;
use wdigital\users\traits\ModuleTrait;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\NotSupportedException;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\db\Transaction;
use yii\helpers\ArrayHelper;
use yii\web\Application as WebApplication;
use yii\web\IdentityInterface;


/**
 * User ActiveRecord model.
 *
 * @property bool $isAdmin
 * @property bool $isBlocked
 * @property bool $isConfirmed
 *
 * Database fields:
 * @property integer $id
 * @property string $email
 * @property string $password_hash
 * @property string $auth_key
 * @property string $confirmed_at
 * @property string $blocked_at
 * @property string $registration_ip
 * @property string $last_login_at
 * @property integer $login_attempts
 * @property string $created_at
 * @property string $updated_at
 *
 * Defined relations:
 * @property Account[]|null $accounts
 * @property Profile|null $profile
 *
 * Dependencies:
 * @property-read Finder $finder
 * @property-read Module $module
 * @property-read mixed $authKey
 * @property-read Mailer $mailer
 */
class User extends ActiveRecord implements IdentityInterface
{
    use ModuleTrait;

    const BEFORE_CREATE = 'beforeCreate';
    const AFTER_CREATE = 'afterCreate';
    const BEFORE_REGISTER = 'beforeRegister';
    const AFTER_REGISTER = 'afterRegister';
    const BEFORE_CONFIRM = 'beforeConfirm';
    const AFTER_CONFIRM = 'afterConfirm';

    /** @var string|null Plain password. Used for model validation. */
    public ?string $password = null;

    /** @var Profile|null */
    private ?Profile $_profile = null;

    /**
     * Handles incorrect login attempt for user record
     */
    public function incorrectLoginAttempt(): void
    {
        if ($this->module->enableBlockingByLoginAttempts) {
            if ($this->module->enableLoginAttemptsForAdmins || !$this->getIsAdmin()) {
                $this->updateAttributes(['login_attempts' => $this->login_attempts + 1]);
            }
            if ($this->login_attempts >= $this->module->maxLoginAttempts) {
                $this->block();
            }
        }
    }

    /**
     * @return Finder
     * @throws InvalidConfigException
     */
    protected function getFinder(): Finder
    {
        return Yii::$container->get(Finder::class);
    }

    /**
     * @return Mailer
     * @throws InvalidConfigException
     */
    protected function getMailer(): Mailer
    {
        return Yii::$container->get(Mailer::class);
    }

    /**
     * @return bool Whether the user is confirmed or not.
     */
    public function getIsConfirmed(): bool
    {
        return $this->confirmed_at !== null;
    }

    /**
     * @return bool Whether the user is blocked or not.
     */
    public function getIsBlocked(): bool
    {
        return $this->blocked_at !== null;
    }

    /**
     * @return bool Whether the user is an admin or not.
     */
    public function getIsAdmin(): bool
    {
        return
            (Yii::$app->getAuthManager() && $this->module->adminPermission ?
                Yii::$app->authManager->checkAccess($this->id, $this->module->adminPermission) : false)
            || in_array($this->email, $this->module->admins, true);
    }

    /**
     * @return ActiveQuery|null
     */
    public function getProfile(): ?ActiveQuery
    {
        if ($this->module->enableUserProfile) {
            return $this->hasOne($this->module->modelMap['Profile'], ['user_id' => 'id']);
        }
        return null;
    }

    /**
     * @param Profile $profile
     */
    public function setProfile(Profile $profile): void
    {
        $this->_profile = $profile;
    }

    /**
     * @return Account[]|null Connected accounts ($provider => $account)
     */
    public function getAccounts(): ?array
    {
        if (!$this->module->enableUserSocialAccount) {
            return null;
        }
        $connected = [];
        $accounts = $this->hasMany($this->module->modelMap['Account'], ['user_id' => 'id'])->all();

        /** @var Account $account */
        foreach ($accounts as $account) {
            $connected[$account->provider] = $account;
        }

        return $connected;
    }

    /**
     * Returns connected account by provider.
     * @param string $provider
     * @return Account|null
     */
    public function getAccountByProvider(string $provider): ?Account
    {
        $accounts = $this->getAccounts();
        return $accounts[$provider] ?? null;
    }

    /** @inheritdoc */
    public function getId()
    {
        return $this->getAttribute('id');
    }

    /** @inheritdoc */
    public function getAuthKey(): ?string
    {
        return $this->getAttribute('auth_key');
    }

    /** @inheritdoc */
    public function attributeLabels(): array
    {
        return [
            'email' => Yii::t('user', 'Email'),
            'registration_ip' => Yii::t('user', 'Registration ip'),
            'unconfirmed_email' => Yii::t('user', 'New email'),
            'password' => Yii::t('user', 'Password'),
            'created_at' => Yii::t('user', 'Registration time'),
            'updated_at' => Yii::t('user', 'Updated at'),
            'last_login_at' => Yii::t('user', 'Last login'),
            'confirmed_at' => Yii::t('user', 'Confirmation time'),
            'blocked_at' => Yii::t('user', 'Blocked at'),
        ];
    }

    /** @inheritdoc */
    public function behaviors(): array
    {
        return [
            [
                'class' => TimestampBehavior::class,
                'value' => (new DateTime())->format('Y-m-d H:i:s')
            ]
        ];
    }

    /** @inheritdoc */
    public function scenarios(): array
    {
        $scenarios = parent::scenarios();
        return ArrayHelper::merge($scenarios, [
            'register' => ['email', 'password'],
            'connect' => ['email'],
            'create' => ['email', 'password'],
            'update' => ['email', 'password'],
        ]);
    }

    /** @inheritdoc */
    public function rules(): array
    {
        return [
            // email rules
            'emailTrim' => ['email', 'trim'],
            'emailRequired' => ['email', 'required', 'on' => ['register', 'connect', 'create', 'update']],
            'emailPattern' => ['email', 'email'],
            'emailLength' => ['email', 'string', 'max' => 255],
            'emailUnique' => [
                'email',
                'unique',
                'message' => Yii::t('user', 'This email address has already been taken')
            ],

            // password rules
            'passwordRequired' => ['password', 'required', 'on' => ['register']],
            'passwordLength' => ['password', 'string', 'min' => 6, 'max' => 72, 'on' => ['register', 'create']],
        ];
    }

    /** @inheritdoc */
    public function validateAuthKey($authKey): bool
    {
        return $this->getAttribute('auth_key') === $authKey;
    }

    /**
     * Creates new user account. If Module::enableGeneratingPassword is set true, this method
     * will generate password.
     *
     * @return bool
     * @throws InvalidConfigException
     * @throws \yii\db\Exception
     */
    public function create(): bool
    {
        if ($this->getIsNewRecord() === false) {
            throw new RuntimeException('Calling "' . __CLASS__ . '::' . __METHOD__ . '" on existing user');
        }

        $transaction = static::getDb()->beginTransaction();
        if (!$transaction instanceof Transaction) {
            throw new RuntimeException('Could not start database transaction for user creation');
        }
        try {
            $this->password = $this->module->enableGeneratingPassword ? Password::generate(12) : $this->password;
            $this->trigger(self::BEFORE_CREATE);
            if (!$this->save()) {
                $transaction->rollBack();
                return false;
            }
            if ($this->module->enableConfirmation) {
                /** @var Token $token */
                $token = Yii::createObject(['class' => Token::class, 'type' => Token::TYPE_CONFIRMATION]);
                $token->link('user', $this);
            }

            $this->mailer->sendWelcomeMessage($this, $token ?? null);

            $this->trigger(self::AFTER_CREATE);
            $transaction->commit();
            return true;
        } catch (Exception $e) {
            $transaction->rollBack();
            Yii::warning($e->getMessage());
            throw $e;
        }
    }

    /**
     * This method is used to register new user account. If Module::enableConfirmation is set true, this method
     * will generate new confirmation token and use mailer to send it to the user.
     *
     * @return bool
     */
    public function register()
    {
        if ($this->getIsNewRecord() == false) {
            throw new RuntimeException('Calling "' . __CLASS__ . '::' . __METHOD__ . '" on existing user');
        }

        $transaction = $this->getDb()->beginTransaction();

        try {
            $this->confirmed_at = $this->module->enableConfirmation ? null : (new DateTime())->format('Y-m-d H:i:s');
            $this->password = $this->module->enableGeneratingPassword ? Password::generate(8) : $this->password;

            $this->trigger(self::BEFORE_REGISTER);

            if (!$this->save()) {
                $transaction->rollBack();
                return false;
            }

            if ($this->module->enableConfirmation) {
                /** @var Token $token */
                $token = Yii::createObject(['class' => Token::className(), 'type' => Token::TYPE_CONFIRMATION]);
                $token->link('user', $this);
            }

            $this->mailer->sendWelcomeMessage($this, isset($token) ? $token : null);
            $this->trigger(self::AFTER_REGISTER);

            $transaction->commit();

            return true;
        } catch (Exception $e) {
            $transaction->rollBack();
            Yii::warning($e->getMessage());
            throw $e;
        }
    }

    /**
     * Attempts user confirmation.
     *
     * @param string $code Confirmation code.
     *
     * @return boolean
     */
    public function attemptConfirmation($code)
    {
        $token = $this->finder->findTokenByParams($this->id, $code, Token::TYPE_CONFIRMATION);

        if ($token instanceof Token && !$token->isExpired) {
            $token->delete();
            if (($success = $this->confirm())) {
                Yii::$app->user->login($this, $this->module->rememberFor);
                $message = Yii::t('user', 'Thank you, registration is now complete.');
            } else {
                $message = Yii::t('user', 'Something went wrong and your account has not been confirmed.');
            }
        } else {
            $success = false;
            $message = Yii::t('user', 'The confirmation link is invalid or expired. Please try requesting a new one.');
        }

        Yii::$app->session->setFlash($success ? 'success' : 'danger', $message);

        return $success;
    }

    /**
     * Generates a new password and sends it to the user.
     *
     * @param string $code Confirmation code.
     *
     * @return boolean
     */
    public function resendPassword()
    {
        $this->password = Password::generate(8);
        $this->save(false, ['password_hash']);

        return $this->mailer->sendGeneratedPassword($this, $this->password);
    }

    /**
     * Confirms the user by setting 'confirmed_at' field to current time.
     */
    public function confirm()
    {
        $this->trigger(self::BEFORE_CONFIRM);
        $result = (bool)$this->updateAttributes(['confirmed_at' => (new DateTime())->format('Y-m-d H:i:s')]);
        $this->trigger(self::AFTER_CONFIRM);
        return $result;
    }

    /**
     * Resets password.
     *
     * @param string $password
     *
     * @return bool
     */
    public function resetPassword($password)
    {
        return (bool)$this->updateAttributes(['password_hash' => Password::hash($password)]);
    }

    /**
     * Blocks the user by setting 'blocked_at' field to current time and regenerates auth_key.
     * @return bool
     * @throws InvalidConfigException
     * @throws \yii\base\Exception
     */
    public function block(): bool
    {
        $this->setAttributes([
            'blocked_at' => (new DateTime())->format('Y-m-d H:i:s'),
            'auth_key' => Yii::$app->security->generateRandomString(),
        ], false);
        if ($this->module->enableBlockingEmailNotification) {
            $this->getMailer()->sendBlockedMessage($this);
        }
        return (bool)$this->updateAttributes(['blocked_at', 'auth_key']);
    }

    /**
     * UnBlocks the user record
     * @throws InvalidConfigException
     */
    public function unblock(): bool
    {
        if ($this->module->enableConfirmation) {
            $token = Yii::createObject(['class' => Token::class, 'type' => Token::TYPE_CONFIRMATION]);
            $token->link('user', $this);
            $this->getMailer()->sendUnblockedMessage($this, $token ?? null);
            return (bool)$this->updateAttributes(['blocked_at' => null, 'confirmed_at' => null, 'login_attempts' => 0]);
        }
        return (bool)$this->updateAttributes(['blocked_at' => null, 'login_attempts' => 0]);
    }

    /** @inheritdoc */
    public function beforeSave($insert)
    {
        if ($insert) {
            $this->setAttribute('auth_key', Yii::$app->security->generateRandomString());
            if (Yii::$app instanceof WebApplication) {
                $this->setAttribute('registration_ip', Yii::$app->request->userIP);
            }
        }

        if (!empty($this->password)) {
            $this->setAttribute('password_hash', Password::hash($this->password));
        }

        return parent::beforeSave($insert);
    }

    /** @inheritdoc */
    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);
        if ($insert) {
            if ($this->_profile == null) {
                $this->_profile = Yii::createObject(Profile::className());
            }
            $this->_profile->link('user', $this);
        }
    }

    /** @inheritdoc */
    public static function tableName()
    {
        return '{{%user}}';
    }

    /** @inheritdoc */
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    /** @inheritdoc */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        throw new NotSupportedException('Method "' . __CLASS__ . '::' . __METHOD__ . '" is not implemented.');
    }
}
