<?php

namespace wdigital\users\controllers;

use Throwable;
use wdigital\users\Finder;
use wdigital\users\models\LoginForm;
use wdigital\users\models\RecoveryForm;
use wdigital\users\models\Token;
use wdigital\users\Module;
use Yii;
use yii\base\InvalidConfigException;
use yii\db\StaleObjectException;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\BadRequestHttpException;
use yii\web\Controller;
use yii\web\HttpException;
use yii\web\NotFoundHttpException;
use yii\web\Response;

/**
 * Class SecurityController
 * Controller that manages user authentication process.
 * @package wdigital\users\controllers
 */
class SecurityController extends Controller
{
    /** @var Finder */
    protected Finder $finder;

    /**
     * @param string $id
     * @param Module $module
     * @param Finder $finder
     * @param array $config
     */
    public function __construct($id, $module, Finder $finder, $config = [])
    {
        $this->finder = $finder;
        parent::__construct($id, $module, $config);
    }

    /** @inheritdoc */
    public function behaviors(): array
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['allow' => true, 'actions' => ['login', 'forgot', 'reset', 'confirm'], 'roles' => ['?']],
                    ['allow' => true, 'actions' => ['login', 'logout'], 'roles' => ['@']],
                ],
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'logout' => ['post'],
                ],
            ],
        ];
    }

    /**
     * @inheritDoc
     * @throws BadRequestHttpException
     */
    public function beforeAction($action): bool
    {
        if ($this->module->unauthenticatedLayout !== null) {
            $this->layout = $this->module->unauthenticatedLayout;
        }
        return parent::beforeAction($action);
    }

    /**
     * Displays the login page.
     *
     * @return string|Response
     * @throws InvalidConfigException
     */
    public function actionLogin()
    {
        if (!Yii::$app->user->isGuest) {
            $this->goHome();
        }
        $model = Yii::createObject(LoginForm::class);
        if ($model->load(Yii::$app->getRequest()->post()) && $model->login()) {
            return $this->goBack();
        }
        return $this->render('login', [
            'model' => $model,
            'module' => $this->module,
        ]);
    }

    /**
     * Logs the user out and then redirects to the homepage.
     *
     * @return Response
     */
    public function actionLogout(): Response
    {
        Yii::$app->getUser()->logout();
        return $this->goHome();
    }

    /**
     * Shows page where user can request password recovery.
     *
     * @return string|Response
     * @throws InvalidConfigException
     * @throws NotFoundHttpException
     */
    public function actionForgot()
    {
        if (!$this->module->enablePasswordRecovery) {
            throw new NotFoundHttpException();
        }
        $model = Yii::createObject([
            'class' => RecoveryForm::class,
            'scenario' => RecoveryForm::SCENARIO_FORGOT,
        ]);
        if ($model->load(Yii::$app->request->post()) && $model->sendRecoveryMessage()) {
            return $this->redirect(['security/login']);
        }
        return $this->render('request', [
            'model' => $model,
            'module' => $this->module,
        ]);
    }

    /**
     * Displays page where user can reset password.
     *
     * @param int|null $id
     * @param string|null $code
     *
     * @return string|Response
     * @throws InvalidConfigException
     * @throws NotFoundHttpException
     * @throws Throwable
     * @throws StaleObjectException
     */
    public function actionReset(?int $id = null, ?string $code = null)
    {
        if (!$this->module->enablePasswordRecovery) {
            throw new NotFoundHttpException();
        }
        $token = $this->finder->findToken(['user_id' => $id, 'code' => $code, 'type' => Token::TYPE_RECOVERY])->one();
        if (!$token instanceof Token) {
            throw new NotFoundHttpException();
        }
        if ($token->isExpired || $token->user === null) {
            throw new BadRequestHttpException(Yii::t('user', 'Recovery link is invalid or expired. Please try requesting a new one.'));
        }
        $model = Yii::createObject([
            'class' => RecoveryForm::class,
            'scenario' => RecoveryForm::SCENARIO_RESET,
        ]);
        if ($model->load(Yii::$app->getRequest()->post()) && $model->resetPassword($token)) {
            return $this->redirect(['security/login']);
        }
        return $this->render('reset', [
            'model' => $model,
            'module' => $this->module,
            'title' => Yii::t('user', 'Reset your password')
        ]);
    }

    /**
     * Confirms user's account. If confirmation was successful logs the user and shows success message. Otherwise
     * shows error message.
     *
     * @param int|null $id
     * @param string|null $code
     *
     * @return string
     * @throws InvalidConfigException
     * @throws NotFoundHttpException
     * @throws StaleObjectException
     * @throws Throwable
     */
    public function actionConfirm(?int $id = null, ?string $code = null)
    {
        if (!$this->module->enableConfirmation) {
            throw new NotFoundHttpException();
        }
        $token = $this->finder->findToken(['user_id' => $id, 'code' => $code, 'type' => Token::TYPE_CONFIRMATION])->one();
        if (!$token instanceof Token) {
            throw new NotFoundHttpException();
        }
        if ($token->isExpired || $token->user === null) {
            throw new BadRequestHttpException(Yii::t('user', 'Recovery link is invalid or expired. Please try requesting a new one.'));
        }

        $model = Yii::createObject([
            'class' => RecoveryForm::class,
            'scenario' => RecoveryForm::SCENARIO_RESET,
        ]);

        if ($model->load(Yii::$app->getRequest()->post()) && $model->confirmPassword($token)) {
            return $this->redirect(['security/login']);
        }

        return $this->render('reset', [
            'model' => $model,
            'module' => $this->module,
            'title' => Yii::t('user', 'Create your password')
        ]);
    }
}
