<?php

namespace wdigital\users\models;

use Throwable;
use wdigital\users\Finder;
use wdigital\users\Mailer;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\db\StaleObjectException;

/**
 * Class RecoveryForm
 *
 * Model for collecting data on password recovery.
 *
 * @package wdigital\users\models
 */
class RecoveryForm extends Model
{
    /**
     * Password recovery request scenario
     */
    public const SCENARIO_FORGOT = 'forgot';

    /**
     * Password reset scenario
     */
    public const SCENARIO_RESET = 'reset';

    /**
     * @var string|null
     */
    public ?string $email = null;

    /**
     * @var string|null
     */
    public ?string $password = null;

    /**
     * @var string|null
     */
    public ?string $password_repeat = null;

    /**
     * @var Mailer
     */
    protected Mailer $mailer;

    /**
     * @var Finder
     */
    protected Finder $finder;

    /**
     * @param Mailer $mailer
     * @param Finder $finder
     * @param array $config
     */
    public function __construct(Mailer $mailer, Finder $finder, $config = [])
    {
        $this->mailer = $mailer;
        $this->finder = $finder;
        parent::__construct($config);
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'email' => Yii::t('user', 'Email'),
            'password' => Yii::t('user', 'Password'),
            'password_repeat' => Yii::t('user', 'Repeat password'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function scenarios(): array
    {
        return [
            self::SCENARIO_FORGOT => ['email'],
            self::SCENARIO_RESET => ['password', 'password_repeat'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            'emailTrim' => ['email', 'trim'],
            'emailRequired' => ['email', 'required'],
            'emailPattern' => ['email', 'email'],
            'passwordRequired' => [['password', 'password_repeat'], 'required'],
            'passwordLength' => ['password', 'string', 'min' => 9, 'max' => 72],
            'passwordCompare' => ['password_repeat', 'compare', 'operator' => '===', 'compareAttribute' => 'password', 'skipOnEmpty' => false, 'skipOnError' => true],
            'passwordMatch' => ['password', 'match', 'pattern' => '/(?=.*[a-z])(?=.*[A-Z])((?=.*\W)|(?=.*\d)).+$/', 'message' => Yii::t('user', '{attribute} should contain one uppercase character, one lowercase character, one digit or symbol'), 'skipOnEmpty' => false, 'skipOnError' => true],
        ];
    }

    /**
     * Sends recovery message.
     *
     * @return bool
     * @throws InvalidConfigException
     */
    public function sendRecoveryMessage(): bool
    {
        if (!$this->validate()) {
            return false;
        }
        $user = $this->finder->findUserByEmail($this->email);
        if ($user instanceof User) {
            $token = Yii::createObject([
                'class' => Token::class,
                'user_id' => $user->id,
                'type' => Token::TYPE_RECOVERY,
            ]);
            if (!$token->save(false)) {
                return false;
            }
            if (!$this->mailer->sendRecoveryMessage($user, $token)) {
                return false;
            }
        }
        Yii::$app->session->setFlash(
            'info',
            Yii::t('user', 'An email has been sent with instructions for resetting your password')
        );
        return true;
    }

    /**
     * Resets user's password.
     *
     * @param Token $token
     *
     * @return bool
     * @throws Throwable
     * @throws StaleObjectException
     */
    public function resetPassword(Token $token): bool
    {
        if (!$this->validate() || $token->user === null) {
            return false;
        }

        if ($token->user->resetPassword($this->password)) {
            Yii::$app->session->setFlash('success', Yii::t('user', 'Your password has been changed successfully.'));
            $token->delete();
        } else {
            Yii::$app->session->setFlash(
                'danger',
                Yii::t('user', 'An error occurred and your password has not been changed. Please try again later.')
            );
        }

        return true;
    }

    /**
     * Resets user's password.
     *
     * @param Token $token
     *
     * @return bool
     * @throws Throwable
     * @throws StaleObjectException
     */
    public function confirmPassword(Token $token): bool
    {
        if (!$this->validate() || $token->user === null) {
            return false;
        }
        $token->delete();
        if ($token->user->resetPassword($this->password) && $token->user->confirm()) {
            Yii::$app->session->setFlash('success', Yii::t('user', 'Thank you, registration is now complete.'));
        } else {
            Yii::$app->session->setFlash('danger',  Yii::t('user', 'Something went wrong and your account has not been confirmed.'));
        }
        return true;
    }
}
