<?php

namespace wdigital\users\models;

use DateTime;
use wdigital\users\Finder;
use wdigital\users\helpers\Password;
use wdigital\users\traits\ModuleTrait;
use Yii;
use yii\base\Model;

/**
 * Class LoginForm
 *
 * LoginForm get user's login and password, validates them and logs the user in. If user has been blocked, it adds
 * an error to login form.
 *
 * @package wdigital\users\models
 */
class LoginForm extends Model
{
    use ModuleTrait;

    /** @var string|null User's email */
    public ?string $login = null;

    /** @var string|null User's plain password */
    public ?string $password = null;

    /** @var bool Whether to remember the user */
    public bool $rememberMe = false;

    /** @var User */
    protected $user;

    /** @var Finder */
    protected Finder $finder;

    /**
     * @param Finder $finder
     * @param array $config
     */
    public function __construct(Finder $finder, $config = [])
    {
        $this->finder = $finder;
        parent::__construct($config);
    }

    /** @inheritdoc */
    public function attributeLabels(): array
    {
        return [
            'login' => Yii::t('user', 'Email'),
            'password' => Yii::t('user', 'Password'),
            'rememberMe' => Yii::t('user', 'Remember me next time'),
        ];
    }

    /** @inheritdoc */
    public function rules(): array
    {
        return [
            'loginTrim' => ['login', 'trim'],
            'requiredFields' => [['login', 'password'], 'required'],
            'passwordValidate' => ['password', 'validatePassword'],
            'confirmationValidate' => ['login', 'validateLogin'],
            'rememberMe' => ['rememberMe', 'boolean'],
        ];
    }

    /**
     * Validates if the hash of the given password is identical to the saved hash in the database.
     *
     * @param string $attribute
     */
    public function validatePassword(string $attribute): void
    {
        if ($this->user !== null) {
            if (!$this->user->hasErrors() && !Password::validate($this->password, $this->user->password_hash)) {
                $this->user->incorrectLoginAttempt();
            } elseif (Password::validate($this->password, $this->user->password_hash)) {
                return;
            }
        }
        $this->addError($attribute, Yii::t('user', 'Invalid login or password'));
    }

    /**
     * Validates if the user trying to log in blocked or not confirmed his email
     *
     * @param string $attribute
     */
    public function validateLogin(string $attribute): void
    {
        if ($this->user !== null) {
            $confirmationRequired = $this->module->enableConfirmation
                && !$this->module->enableUnconfirmedLogin;
            if ($confirmationRequired && !$this->user->getIsConfirmed()) {
                $this->addError($attribute, Yii::t('user', 'You need to confirm your email address'));
            }
            if ($this->user->getIsBlocked()) {
                $this->addError($attribute, Yii::t('user', 'Your account has been blocked'));
            }
        }
    }

    /**
     * Validates form and logs the user in.
     *
     * @return bool whether the user is logged in successfully
     */
    public function login(): bool
    {
        if ($this->validate() && $this->user) {
            $isLogged = Yii::$app->getUser()->login($this->user, $this->rememberMe ? $this->module->rememberFor : 0);

            if ($isLogged) {
                $this->user->updateAttributes(['last_login_at' => (new DateTime())->format('Y-m-d H:i:s'), 'login_attempts' => 0]);
            }

            return $isLogged;
        }

        return false;
    }

    /** @inheritdoc */
    public function beforeValidate(): bool
    {
        if (parent::beforeValidate()) {
            $this->user = $this->finder->findUserByEmail(trim($this->login));
            return true;
        }
        return false;
    }
}
