<?php


namespace wdigital\users\controllers;

use Yii;
use yii\base\InvalidConfigException;
use yii\di\Instance;
use yii\filters\VerbFilter;
use yii\rbac\ManagerInterface;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\Response;
use yii\widgets\ActiveForm;

/**
 * Controller for managing rules.
 *
 * @author Dmitry Erofeev <dmeroff@gmail.com>
 */
class RuleController extends Controller
{
    /**
     * @var ManagerInterface The auth manager component.
     */
    public ManagerInterface $authManager;

    /**
     * This method will set [[authManager]] to be the 'authManager' application component, if it is `null`.
     */
    public function init(): void
    {
        $this->authManager = Yii::$app->getAuthManager();
        parent::init();
    }

    /**
     * @inheritDoc
     */
    public function behaviors(): array
    {
        return [
            'verbFilter' => [
                'class'   => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
        ];
    }

    /**
     * Shows list of created rules.
     *
     * @return string
     * @throws InvalidConfigException
     */
    public function actionIndex()
    {
        $searchModel  = $this->getSearchModel();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel'  => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Shows page where new rule can be added.
     *
     * @return array|string
     */
    public function actionCreate()
    {
        $model = $this->getModel(Rule::SCENARIO_CREATE);

        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if ($model->load(Yii::$app->request->post()) && $model->create()) {
            Yii::$app->session->setFlash('success', Yii::t('rbac', 'Rule has been added'));
            return $this->redirect(['index']);
        }

        return $this->render('create', [
            'model' => $model,
        ]);
    }

    /**
     * Updates existing auth rule.
     * @param  string $name
     * @return array|string|Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($name)
    {
        $model = $this->getModel(Rule::SCENARIO_UPDATE);
        $rule  = $this->findRule($name);

        $model->setOldName($name);
        $model->setAttributes([
            'name'  => $rule->name,
            'class' => get_class($rule),
        ]);

        if (Yii::$app->request->isAjax && $model->load(Yii::$app->request->post())) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            return ActiveForm::validate($model);
        }

        if ($model->load(Yii::$app->request->post()) && $model->update()) {
            Yii::$app->session->setFlash('success', Yii::t('rbac', 'Rule has been updated'));
            return $this->redirect(['index']);
        }

        return $this->render('update', [
            'model' => $model,
        ]);
    }

    /**
     * Removes rule.
     *
     * @param  string $name
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionDelete($name)
    {
        $rule = $this->findRule($name);

        $this->authManager->remove($rule);
        $this->authManager->invalidateCache();

        Yii::$app->session->setFlash('success', Yii::t('rbac', 'Rule has been removed'));

        return $this->redirect(['index']);
    }

    /**
     * Searches for rules.
     *
     * @param  string|null $q
     * @return array
     */
    public function actionSearch($q = null)
    {
        Yii::$app->response->format = Response::FORMAT_JSON;

        return ['results' => $this->getSearchModel()->getRuleNames($q)];
    }

    /**
     * @param  string $scenario
     * @return Rule
     * @throws InvalidConfigException
     */
    private function getModel($scenario)
    {
        return Yii::createObject([
            'class'    => Rule::className(),
            'scenario' => $scenario,
        ]);
    }

    /**
     * @return RuleSearch
     * @throws InvalidConfigException
     */
    private function getSearchModel()
    {
        return Yii::createObject(RuleSearch::className());
    }

    /**
     * @param  string $name
     * @return mixed|null|\yii\rbac\Rule
     * @throws NotFoundHttpException
     */
    private function findRule($name)
    {
        $rule = $this->authManager->getRule($name);

        if ($rule instanceof \yii\rbac\Rule) {
            return $rule;
        }

        throw new NotFoundHttpException(Yii::t('rbac', 'Not found'));
    }
}
