<?php

namespace wdigital\cms\user\models;

use dektrium\user\models\Token;
use wdigital\cms\user\helpers\Password;
use dektrium\user\models\User as Model;
use wdigital\cms\user\Module;
use Yii;

/**
 * @property-read Module $module
 */
class User extends Model
{
    const SCENARIO_REGISTER = 'register';
    const SCENARIO_CONNECT = 'connect';
    const SCENARIO_CREATE = 'create';
    const SCENARIO_UPDATE = 'update';
    const SCENARIO_SETTINGS = 'settings';

    /**
     * @inheritdoc
     */
    public function rules()
    {
        $rules = parent::rules();

        if (!$this->module->enableUsernames) {
            unset($rules['usernameRequired'], $rules['usernameMatch']);
            $rules = [
                'usernameDefault' => [
                    'username', 'default', 'value' => function ($model) {
                        return $model->email;
                    },
                ],
            ] + $rules;
        }

        $rules['passwordLength'] = ['password', 'string', 'min' => 9, 'max' => 72, 'on' => ['register', 'create']];
        $rules['passwordMatch'] = ['password', 'match', 'pattern' => '/(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*\W).+$/', 'message' => Yii::t('user', '{attribute} should contain one uppercase character, lowercase character, digit and symbol')];

        return $rules;
    }

    /**
     * Creates new user account. It generates password if it is not provided by user.
     *
     * @return bool
     */
    public function create()
    {
        if ($this->getIsNewRecord() == false) {
            throw new \RuntimeException('Calling "' . __CLASS__ . '::' . __METHOD__ . '" on existing user');
        }

        $transaction = $this->getDb()->beginTransaction();

        try {
            $this->password = $this->password == null ? Password::generate(12) : $this->password;

            $this->trigger(self::BEFORE_CREATE);

            if (!$this->save()) {
                $transaction->rollBack();
                return false;
            }

            $this->confirm();

            $this->mailer->sendWelcomeMessage($this, null, true);
            $this->trigger(self::AFTER_CREATE);

            $transaction->commit();

            return true;
        } catch (\Exception $e) {
            $transaction->rollBack();
            \Yii::warning($e->getMessage());
            throw $e;
        }
    }

    /**
     * This method is used to register new user account. If Module::enableConfirmation is set true, this method
     * will generate new confirmation token and use mailer to send it to the user.
     *
     * @return bool
     */
    public function register()
    {
        if ($this->getIsNewRecord() == false) {
            throw new \RuntimeException('Calling "' . __CLASS__ . '::' . __METHOD__ . '" on existing user');
        }

        $transaction = $this->getDb()->beginTransaction();

        try {
            $this->confirmed_at = $this->module->enableConfirmation ? null : time();
            $this->password = $this->module->enableGeneratingPassword ? Password::generate(12) : $this->password;

            $this->trigger(self::BEFORE_REGISTER);

            if (!$this->save()) {
                $transaction->rollBack();
                return false;
            }

            if ($this->module->enableConfirmation) {
                /** @var Token $token */
                $token = \Yii::createObject(['class' => Token::className(), 'type' => Token::TYPE_CONFIRMATION]);
                $token->link('user', $this);
            }

            $this->mailer->sendWelcomeMessage($this, isset($token) ? $token : null);
            $this->trigger(self::AFTER_REGISTER);

            $transaction->commit();

            return true;
        } catch (\Exception $e) {
            $transaction->rollBack();
            \Yii::warning($e->getMessage());
            throw $e;
        }
    }

    /**
     * Generates a new password and sends it to the user.
     *
     * @param string $code Confirmation code.
     *
     * @return boolean
     */
    public function resendPassword()
    {
        $this->password = Password::generate(12);
        $this->save(false, ['password_hash']);

        return $this->mailer->sendGeneratedPassword($this, $this->password);
    }
}
