<?php

namespace wdigital\cms\user;

use dektrium\user\Module as BaseModule;
use wdigital\cms\user\models\LoginForm;
use wdigital\cms\user\models\RecoveryForm;
use wdigital\cms\user\models\RegistrationForm;
use wdigital\cms\user\models\SettingsForm;
use wdigital\cms\user\models\User;
use wdigital\inheritingMessageSource\InheritingMessageSourceInterface;
use wdigital\inheritingMessageSource\PhpMessageSource as InheritingPhpMessageSource;
use Yii;
use yii\console\Application as ConsoleApplication;
use yii\helpers\ArrayHelper;
use yii\helpers\StringHelper;
use yii\i18n\PhpMessageSource;
use yii\web\Application as WebApplication;

class Module extends BaseModule
{
    /**
     * @inheritdoc
     */
    public $controllerNamespace = 'dektrium\\user\\controllers';

    /**
     * Console application command (controller ID) for running migrations.
     * @var string
     */
    public $migrateCommand = 'migrate';

    /**
     * Location of the migrations for this module.
     * @var string
     */
    public $migrationPath = '@vendor/dektrium/yii2-user/migrations';

    /**
     * The namespace of migrations for this module.
     * @var string
     */
    public $migrationNamespace = __NAMESPACE__ . '\\migrations';

    /**
     * Prompt the user for a new password twice to verify the ability to replicate the password.
     * @var bool
     */
    public $repeatPasswordPrompt = true;

    /**
     * Whether usernames can be chosen/assigned and used for logging in.
     * @var bool
     */
    public $enableUsernames = false;

    /**
     * @inheritdoc
     */
    public $admins = ['admin'];

    /**
     * Points to the parent module views.
     * @var string
     */
    protected $defaultViewPath = '@vendor/dektrium/yii2-user/views';

    /**
     * Models overriding the default model map ones.
     * @var array
     */
    protected $overrideModels = [
        LoginForm::class,
        RecoveryForm::class,
        RegistrationForm::class,
        SettingsForm::class,
        User::class,
    ];

    /**
     * @inheritdoc
     */
    public function __construct($id, $parent = null, $config = [])
    {
        if (empty($config['viewPath'])) {
            $config['viewPath'] = $this->defaultViewPath;
        }
        if (empty($config['modelMap'])) {
            $config['modelMap'] = [];
        }
        $config['modelMap'] += $this->modelMapOverrides();

        parent::__construct($id, $parent, $config);
    }

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();

        $app = Yii::$app;
        if ($app instanceof WebApplication) {
            $this->injectTranslations($app);
            $this->overrideViews($app);
        } elseif ($app instanceof ConsoleApplication) {
            $this->injectMigrations($app);
        }
    }

    /**
     * @param WebApplication $app
     */
    private function overrideViews($app)
    {
        if (empty($app->view->theme)) {
            $app->view->theme = Yii::createObject(['class' => 'yii\base\Theme']);
        }
        $pathMap = $app->view->theme->pathMap;

        $defaultViewPath = $this->defaultViewPath;
        if (empty($pathMap[$defaultViewPath])) {
            $pathMap[$defaultViewPath] = [];
        } elseif (is_string($pathMap[$defaultViewPath])) {
            $pathMap[$defaultViewPath] = [$pathMap[$defaultViewPath]];
        }
        $pathMap[$defaultViewPath][] = '@vendor/wdigital/cms-user/src/views';

        $app->view->theme->pathMap = $pathMap;
    }

    /**
     * @return array
     */
    private function modelMapOverrides()
    {
        $modelMap = [];
        foreach ($this->overrideModels as $model => $override) {
            if (is_integer($model)) {
                $model = StringHelper::basename($override);
            }
            $modelMap[$model] = $override;
        }

        return $modelMap;
    }

    /**
     * Called when bootstrapping a console application with this module.
     *
     * @param ConsoleApplication $app
     */
    private function injectMigrations($app)
    {
        $migrateCommand = $this->migrateCommand;

        $migrateConfig = $this->retrieveCommandConfig($app, $migrateCommand);
        if ($migrateConfig) {
            $migrateConfig = is_string($migrateConfig)
                ? ['class' => $migrateConfig]
                : (array)$migrateConfig; // better safe than sorry

            $app->controllerMap[$migrateCommand] = $this->setSubValue($migrateConfig, 'migrationPath', $this->migrationPath, ['@app/migrations']);
            $app->controllerMap[$migrateCommand] = $this->setSubValue($app->controllerMap[$migrateCommand], 'migrationNamespaces', $this->migrationNamespace);
        }
    }

    /**
     * @param WebApplication|ConsoleApplication $app
     */
    protected function injectTranslations($app)
    {
        $messageSources = &$app->i18n->translations;

        if (empty($messageSources['user*'])) {
            $parentSource = [
                'class' => PhpMessageSource::class,
                'basePath' => '@vendor/dektrium/yii2-user/messages',
                'sourceLanguage' => 'en-US',
            ];
        } else {
            $parentSource = $messageSources['user*'];
            if (is_array($parentSource) || is_string($parentSource)) {
                $parentSource = $messageSources['user*'] = \Yii::createObject($parentSource);
            }
        }

        if (!$parentSource instanceof InheritingMessageSourceInterface) {
            $messageSources['user*'] = [
                'class' => InheritingPhpMessageSource::class,
                'basePath' => __DIR__ . '/messages',
                'sourceLanguage' => 'en-US',
                'parentSources' => [$parentSource],
            ];
        }
    }

    /**
     * Returns the config of a command or false, if no such command is configured.
     *
     * @param ConsoleApplication $app
     * @param string $command
     * @return array|boolean
     */
    protected function retrieveCommandConfig($app, $command)
    {
        if (!empty($app->controllerMap[$command])) {
            return $app->controllerMap[$command];
        } else {
            $coreCommands = $app->coreCommands();
            if (isset($coreCommands[$command])) {
                return $coreCommands[$command];
            }
        }

        return false;
    }

    /**
     * Inject a sub-value into an array element of an array.
     *
     * @param array $array
     * @param string $parentPath Path to the parent element.
     * @param mixed $value
     * @param array $parentDefault Default value for the parent element.
     * @param string $key Key in the parent element to set. If null, the value will be appended to the parent.
     * @return array The modified array.
     */
    private function setSubValue($array, $parentPath, $value, $parentDefault = [], $key = null)
    {
        $parent = ArrayHelper::getValue($array, $parentPath, $parentDefault);

        if ($key !== null) {
            $parent[$key] = $value;
        } elseif (!in_array($value, $parent)) {
            $parent[] = $value;
        }

        ArrayHelper::setValue($array, $parentPath, $parent);

        return $array;
    }
}
