<?php

namespace wdigital\cms\siteTree\models;

use Throwable;
use wdigital\cms\siteTree\components\ContentTypeClassResolver;
use wdigital\cms\siteTree\interfaces\ContentTypeInterface;
use wdigital\cms\siteTree\Module;
use Yii;
use yii\base\InvalidConfigException;
use yii\behaviors\SluggableBehavior;
use yii\caching\TagDependency;
use yii\db\ActiveQuery;

/**
 * This is the model class for table "section".
 *
 * @property int $id
 * @property int $parent_id
 * @property int $language_id
 * @property string $title
 * @property string $slug
 * @property string $type
 * @property string $meta_title
 * @property string $meta_description
 * @property string $redirect_to
 * @property string $image
 * @property int $order_by
 * @property int $active
 * @property int $visible
 *
 * @property-read ActiveQuery $childSections
 * @property-read ActiveQuery $html
 * @property MultipleTreeSection $parent
 * @method getVariationModel(mixed $id)
 * @method typecastAttributes()
 */
class MultipleTreeSection extends TreeSection
{
    public const IMG_DIR = 'uploads/sections';

    public array $children;

    public string $fullSlug;

    /**
     * @inheritDoc
     */
    public function behaviors(): array
    {
        return [
            [
                'class' => SluggableBehavior::class,
                'attribute' => 'title',
                'ensureUnique' => true,
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%section}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules(): array
    {
        return [
            [['parent_id', 'language_id', 'order_by'], 'integer'],
            [['active', 'visible'], 'boolean'],
            [['language_id', 'title', 'slug', 'type'], 'required'],
            [['title', 'slug', 'type', 'redirect_to', 'image'], 'string', 'max' => 255],
            [['meta_title'], 'string', 'max' => 60],
            [['meta_description'], 'string', 'max' => 150],
            [['language_id'], 'exist', 'skipOnError' => true, 'targetClass' => Yii::$app->getModule('language')->languageClass, 'targetAttribute' => ['language_id' => 'id']],
            [['parent_id'], 'exist', 'skipOnError' => true, 'targetClass' => static::class, 'targetAttribute' => ['parent_id' => 'id']],
        ];
    }

    /**
     * @return array
     */
    public function attributeLabels(): array
    {
        return [
            'parent_id' => Yii::t('site-tree', 'Parent section'),
            'type' => Yii::t('site-tree', 'Section type'),
            'title' => Yii::t('site-tree', 'Title'),
            'slug' => Yii::t('site-tree', 'Slug'),
            'meta_title' => Yii::t('site-tree', 'Meta title'),
            'meta_description' => Yii::t('site-tree', 'Meta description'),
            'redirect_to' => Yii::t('site-tree', 'Redirect to'),
            'active' => Yii::t('site-tree', 'Active'),
            'visible' => Yii::t('site-tree', 'Visible'),
            'image' => Yii::t('site-tree', 'Image'),
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getParent(): ActiveQuery
    {
        return $this->hasOne(static::class, ['id' => 'parent_id'])->from(['parent' => static::tableName()]);
    }

    /**
     * @return ActiveQuery
     */
    public function getHtml(): ActiveQuery
    {
        return $this->hasMany(Html::class, ['section_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getChildSections(): ActiveQuery
    {
        return $this->hasMany(static::class, ['parent_id' => 'id'])->from(['child' => static::tableName()]);
    }

    /**
     * @param array $array
     */
    public static function generateSiteTreeStructure(array &$array): void
    {
        $module = Yii::$app->getModule('language');
        foreach ($array as &$section) {
            foreach ($array as &$subsection) {
                if ($subsection->parent_id === $section->id) {
                    $section->children[] = $subsection;
                }
            }
        }
        foreach ($array as $key => $s) {
            if ($s->parent_id !== null) {
                unset($array[$key]);
            }
        }
        foreach ($array as &$section) {
            $section->setSlugs($module);
        }
    }

    /**
     * @return bool
     */
    public function beforeValidate(): bool
    {
        if ((int)$this->parent_id === 0) {
            $this->parent_id = null;
        }
        $this->language_id = Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language);
        return parent::beforeValidate();
    }

    /**
     * @param bool $insert
     * @return bool
     */
    public function beforeSave($insert): bool
    {
        if ($insert) {
            $lastSection = self::find()->orderBy('order_by DESC')->limit(1)->one();
            if ($lastSection) {
                $this->order_by = $lastSection->order_by + 1;
            } else {
                $this->order_by = 1;
            }
        }
        return parent::beforeSave($insert);
    }

    /**
     * @param bool $insert
     * @param array $changedAttributes
     */
    public function afterSave($insert, $changedAttributes): void
    {
        if ($this->type === 'html') {
            if (!$insert) {
                $html = Html::find()->where(['section_id' => $this->id])->limit(1)->one();
                if (!$html) {
                    $html = new Html();
                }
            } else {
                $html = new Html();
            }
            $html->section_id = $this->id;
            $html->save();
        }
        if (Yii::$app->hasModule('audit')) {
            if ($insert) {
                Yii::$app->getModule('audit')?->logPost('Lapas koks', "Lapas koka sadaļa: $this->title, izveidota!");
            } else {
                Yii::$app->getModule('audit')?->logPost('Lapas koks', "Lapas koka sadaļa: $this->title (id: $this->id), labota!");
            }
        }
        if (Yii::$app->has('cache')) {
            TagDependency::invalidate(Yii::$app->cache, static::class);
        }
        parent::afterSave($insert, $changedAttributes);
    }

    /**
     * @return false|int
     * @throws Throwable
     */
    public function delete(): bool|int
    {
        foreach ($this->childSections as $child) {
            $child->delete();
        }
        $module = Module::getInstance();
        $contentTypes = $module?->content_types;
        foreach ($contentTypes as $contentType) {
            if (($contentType['value'] === $this->type) && isset($contentType['class'])) {
                (new $contentType['class']())->delete();
            }
        }
        return parent::delete();
    }

    public function afterDelete(): void
    {
        if (Yii::$app->hasModule('audit')) {
            Yii::$app->getModule('audit')?->log('Lapas koks', "Lapas koka sadaļa: $this->title (id: $this->id), dzēsta!");
        }
        if (Yii::$app->has('cache')) {
            TagDependency::invalidate(Yii::$app->cache, static::class);
        }
    }

    /**
     * @param $section
     * @param $post
     * @return bool|null
     */
    public function changeOrder($section, $post): ?bool
    {
        $model = new self();
        $updatedSection = $section;
        if ($post['new_parent'] === '#') {
            $updatedSection->parent_id = null;
        } else {
            $updatedSection->parent_id = $post['new_parent'];
        }
        if ($post['new_parent'] === '#') {
            $sections = $model::find()
                ->where(['parent_id' => NULL])
                ->andWhere(['language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language)])
                ->orderBy('order_by ASC')
                ->all();
        } else {
            $sections = $model::find()->where(['parent_id' => $post['new_parent']])
                ->andWhere(['language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language)])
                ->orderBy('order_by asc')->all();
        }
        $sectionCount = count($sections);
        if ($post['old_position'] > $post['position']) {
            for ($i = 0; $i < $sectionCount; $i++) {
                if ($sections[$i]->id !== (int)$post['node_id']) {
                    if ($i <= $post['old_position'] && $i >= $post['position']) {
                        $sections[$i]->order_by = $i + 1;
                    } else {
                        $sections[$i]->order_by = $i;
                    }
                    $sections[$i]->save();
                }
            }
        } else {
            for ($i = 0; $i < $sectionCount; $i++) {
                if ($sections[$i]->id !== (int)$post['node_id']) {
                    if ($i >= $post['old_position'] && $i <= $post['position']) {
                        $sections[$i]->order_by = $i - 1;
                    } else {
                        $sections[$i]->order_by = $i;
                    }
                    $sections[$i]->save();
                }
            }
        }
        $updatedSection->order_by = $post['position'];
        if ($updatedSection->save()) {
            return true;
        }
        return false;
    }

    /**
     * @param array $modelArray
     * @param array $constraints
     * @param string $relation
     * @return bool|MultipleTreeSection
     */
    public static function findInTree(array $modelArray, array $constraints, string $relation = 'children'): bool|MultipleTreeSection
    {
        foreach ($modelArray as $model) {
            $section = $model->findFromConstraints($constraints, $relation);
            if ($section) {
                return $section;
            }
        }
        return false;
    }

    /**
     * @param array $constraints
     * @param string $relation
     * @return bool|SingleTreeSection|MultipleTreeSection
     */
    protected function findFromConstraints(array $constraints, string $relation): bool|SingleTreeSection|MultipleTreeSection
    {
        if ($this->{$relation}) {
            foreach ($this->{$relation} as $child) {
                $section = $child->findInTree($this->{$relation}, $constraints);
                if ($section) {
                    return $section;
                }
            }
        }
        foreach ($constraints as $field => $constraint) {
            if ($constraint === '') {
                if (!$this->$field) {
                    $section = $this;
                } else {
                    $section = false;
                    break;
                }
            } elseif ($this->$field && $this->$field === $constraint) {
                $section = $this;
            } else {
                $section = false;
                break;
            }
        }
        if (isset($section) && $section) {
            return $section;
        }
        return false;
    }

    /**
     * @param string $type
     * @return bool|SingleTreeSection|MultipleTreeSection
     */
    public function getContentTypeSection(string $type): bool|SingleTreeSection|MultipleTreeSection
    {
        return self::findInTree(Yii::$app->view->params['site-tree'], ['type' => $type]);
    }

    /**
     * @param string $slugs
     * @return array|bool
     * @throws InvalidConfigException
     */
    public static function renderView(string $slugs): bool|array
    {
        $contentTypeSlug = '';
        $section = static::findInTree(Yii::$app->view->params['site-tree'], ['fullSlug' => $slugs]);
        while (!($section && $section->type)) {
            $slugs = explode('/', $slugs);
            $contentTypeSlug = '/' . array_pop($slugs) . $contentTypeSlug;
            if (!$slugs) {
                break;
            }
            $slugs = implode('/', $slugs);
            $section = static::findInTree(Yii::$app->view->params['site-tree'], ['fullSlug' => $slugs]);
        }
        if ($section && $section->type) {
            $contentTypeResolver = new ContentTypeClassResolver();
            if ($section->parent_id) {
                $parentSection = static::findInTree(Yii::$app->view->params['site-tree'], ['id' => $section->parent_id]);
            } else {
                $parentSection = null;
            }
            Yii::$app->view->params['activeSection'] = $section;
            $classInstance = $contentTypeResolver->getContentTypeClassInstance($section->type);
            if ($classInstance instanceof ContentTypeInterface) {
                $viewContent = $contentTypeSlug ? $classInstance->getDetailedFrontendContent($section, $contentTypeSlug) : $classInstance->getFrontendContent($section);
                $view = $contentTypeSlug ? $classInstance->getContentTypeDetailedView($viewContent) : $classInstance->getContentTypeView();
                return [
                    'view' => $view,
                    'section' => $section,
                    'parentSection' => $parentSection,
                    'content' => $viewContent,
                    'contentTypeSlug' => $contentTypeSlug
                ];
            }
            throw new InvalidConfigException('Specified class instance doesn\'t implement ContentTypeInterface');

        }
        return false;
    }

    /**
     * @param int $id
     * @return array
     */
    public static function buildSiteTree(mixed $parentQuery = null): array
    {
        if (!$parentQuery) {
            $parentQuery = static::find()
                ->select(['id', 'parent_id AS p', 'title AS text', 'slug AS s', 'order_by AS o', 'type AS t'])
                ->where([
                    'language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language),
                    'parent_id' => NULL
                ])->orderBy(['p' => SORT_ASC, 'o' => SORT_ASC])
                ->asArray();
        }
        $queried = $parentQuery->all();
        $siteTree = [];
        $callback = function ($id, $text, $slug, $orderby, $type, $parent) {
            return [
                'id' => $id,
                'text' => $text,
                'data' => [
                    'p' => $parent,
                    's' => $slug,
                    'o' => $orderby,
                    't' => $type,
                ]
            ];
        };
        static::buildLevel($queried, array_column($queried, 'id'), $siteTree, $callback);
        return $siteTree['firstLevel']['children'] ?? [];
    }

    /**
     * @param array $newValues
     * @param array $ids
     * @param array $siteTree
     * @param callable $callback
     * @return array
     */
    public static function buildLevel(array $newValues, array $ids, array &$siteTree, callable $callback): array
    {
        foreach ($newValues as $section) {
            if ($section['p']) {
                $siteTree[$section['id']] = &$siteTree[$section['p']]['children'][];
            } else {
                $siteTree[$section['id']] = &$siteTree['firstLevel']['children'][];
            }
            $siteTree[$section['id']] = $callback((int)$section['id'], $section['text'], $section['s'], (int)$section['o'], $section['t'], (int)$section['p']);
        }
        $newValues = static::find()
            ->select(['id', 'parent_id AS p', 'title AS text', 'slug AS s', 'order_by AS o', 'type AS t'])
            ->where([
                'language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language)
            ])
            ->andWhere(['in', 'parent_id', $ids])
            ->orderBy(['p' => SORT_ASC, 'o' => SORT_ASC])
            ->asArray()->all();
        if ($newValues) {
            $ids = array_column($newValues, 'id');
            self::buildLevel($newValues, $ids, $siteTree, $callback);
        }
        return $siteTree;
    }
}
