<?php

namespace wdigital\cms\siteTree\models;

use Throwable;
use wdigital\cms\i18new\components\ActiveRecordHelper;
use wdigital\cms\siteTree\Module;
use Yii;
use yii\base\Model;
use yii\db\ActiveQuery;
use yii\helpers\ArrayHelper;
use yii2tech\ar\variation\VariationBehavior;

/**
 * This is the model class for table "section".
 *
 * @property int $id
 * @property int $parent_id
 * @property string $type
 * @property int $active
 * @property int $visible
 * @property int $order_by
 * @property string $image
 *
 * @property-read ActiveQuery $defaultTranslation
 * @property-read ActiveQuery $parent
 * @property SingleTreeSectionTranslation[] $sectionTranslations
 * @property mixed|null language_id
 * @property array $variationModels
 * @property mixed|null slug
 * @property mixed|null $title
 * @method getVariationModel(mixed $id)
 * @method hasDefaultVariationRelation()
 * @method getVariationModels()
 * @method setVariationModels(array $translationModels)
 */
class SingleTreeSection extends TreeSection
{
    public const IMG_DIR = 'uploads/sections';

    public array $children;

    public string $fullSlug;

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%section}}';
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            [['parent_id', 'order_by'], 'integer'],
            [['type'], 'required'],
            [['type'], 'string', 'max' => 255],
            [['active', 'visible'], 'boolean'],
            ['parent_id', 'filter', 'skipOnError' => true, 'filter' => function ($value) {
                if ((int)$value === 0) {
                    return null;
                }
                return $value;
            }],
            [['parent_id'], 'exist', 'skipOnEmpty' => true, 'targetClass' => static::class, 'targetAttribute' => ['parent_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'id' => Yii::t('site-tree', 'ID'),
            'parent_id' => Yii::t('site-tree', 'Parent section'),
            'type' => Yii::t('site-tree', 'Section type'),
            'active' => Yii::t('site-tree', 'Active section'),
            'visible' => Yii::t('site-tree', 'Visible section'),
            'order_by' => Yii::t('site-tree', 'Order By'),
            'image' => Yii::t('site-tree', 'Image'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function behaviors(): array
    {
        return [
            'translations' => [
                'class' => VariationBehavior::class,
                'variationsRelation' => 'sectionTranslations',
                'defaultVariationRelation' => 'defaultTranslation',
                'variationOptionReferenceAttribute' => 'language_id',
                'optionModelClass' => Yii::$app->getModule('language')->languageClass,
                'defaultVariationOptionReference' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language),
                'variationSaveFilter' => Yii::$app->hasModule('language') ?
                    [ActiveRecordHelper::class, 'filterVariationSave'] : null,
            ],
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getSectionTranslations(): ActiveQuery
    {
        return $this->hasMany(SingleTreeSectionTranslation::class, ['section_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getDefaultTranslation(): ActiveQuery
    {
        return $this->hasDefaultVariationRelation();
    }

    /**
     * @return ActiveQuery
     */
    public function getParent(): ActiveQuery
    {
        return $this->hasOne(static::class, ['id' => 'parent_id'])->from(['parent' => static::tableName()]);
    }

    /**
     * @return ActiveQuery
     */
    public function getChildren(): ActiveQuery
    {
        return $this->hasMany(static::class, ['parent_id' => 'id']);
    }

    /**
     * @param array $sections
     * @param null $parentSlug
     * @return array
     */
    public function setChildren(array &$sections, $parentSlug = null): array
    {
        $id = $this->id;
        $children = &$this->children;
        if (!$parentSlug) {
            $module = Yii::$app->getModule('language');
            $this->fullSlug = '/' . $module?->languageClass::getCodeFromId($this->language_id) . '/' . $this->slug;
        } else {
            $this->fullSlug = $parentSlug . '/' . $this->slug;
        }
        foreach ($sections as $index => $section) {
            if ($section->parent_id === $id) {
                unset($sections[$index]);
                $section->setChildren($sections, $this->fullSlug);
                $children[] = $section;
            }
        }
        return $sections;
    }

    /**
     * @param array $array
     */
    public static function generateSiteTreeStructure(array &$array): void
    {
        foreach ($array as &$section) {
            $section->setChildren($array);
        }
        foreach ($array as $key => $s) {
            if ($s->parent_id !== null) {
                unset($array[$key]);
            }
        }
    }

    /**
     * @param array $data
     * @param null $formName
     * @return bool
     */
    public function load($data, $formName = null): bool
    {
        $translationModels = ArrayHelper::map(
            $this->getVariationModels(),
            'language_id',
            function ($translatedModel) {
                return $translatedModel;
            }
        );
        return parent::load($data, $formName) && Model::loadMultiple($translationModels, $data) && $this->setVariationModels($translationModels) ;
    }

    /**
     * @inheritdoc
     */
    public function beforeValidate(): bool
    {
        if ((int)$this->parent_id === 0) {
            $this->parent_id = null;
        }
        return parent::beforeValidate();
    }

    /**
     * @param bool $insert
     * @return bool
     */
    public function beforeSave($insert): bool
    {
        if ($insert) {
            $lastSection = self::find()->orderBy('order_by DESC')->limit(1)->one();
            if ($lastSection) {
                $this->order_by = $lastSection->order_by + 1;
            } else {
                $this->order_by = 1;
            }
        }
        return parent::beforeSave($insert);
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes): void
    {
        if ($this->type === 'html') {
            if (!$insert) {
                $html = TranslatableHtml::find()->where(['section_id' => $this->id])->limit(1)->one();
                if (!$html) {
                    $html = new TranslatableHtml();
                    $html->getVariationModels();
                }
            } else {
                $html = new TranslatableHtml();
                $html->getVariationModels();
            }
            $html->section_id = $this->id;
            $html->save();

        }
        parent::afterSave($insert, $changedAttributes);
    }

    /**
     * @return false|int
     * @throws Throwable
     */
    public function delete(): bool|int
    {
        $module = Module::getInstance();
        $contentTypes = $module?->content_types;
        foreach ($contentTypes as $contentType) {
            if ($contentType['value'] === $this->type) {
                (new $contentType['class']())->delete();
            }
        }
        return parent::delete();
    }

    /**
     * @param $section
     * @param $post
     * @return bool|null
     */
    public function changeOrder($section, $post): bool|null
    {
        $model = new self();
        $updatedSection = $section;
        if ($post['new_parent'] === '#') {
            $updatedSection->parent_id = null;
        } else {
            $updatedSection->parent_id = $post['new_parent'];
        }
        if ($post['new_parent'] === '#') {
            $sections = $model::find()
                ->where(['parent_id' => NULL])
                ->orderBy('order_by ASC')
                ->all();
            if ($post['old_position'] > $post['position']) {
                foreach ($sections as $s) {
                    if (($s->id !== (int)$post['node_id']) && $s->order_by >= $post['position']) {
                        ++$s->order_by;
                        $s->save();
                    }
                }
            } else {
                $count = 0;
                foreach ($sections as $s) {
                    if ($s->id !== (int) $post['node_id']) {
                        if ($count + 1 <= $post['position']) {
                            $s->order_by = $count;
                            $s->save();
                        }
                        ++$count;
                    }
                }
            }
            $updatedSection->order_by = $post['position'];
        } else {
            $sections = $model::find()->where(['parent_id' => $post['new_parent']])
                ->orderBy('order_by asc')->all();
            if ($post['old_position'] > $post['position']) {
                foreach ($sections as $s) {
                    if (($s->id !== (int)$post['node_id']) && $s->order_by >= $post['position']) {
                        ++$s->order_by;
                        $s->save();
                    }
                }
            } else {
                foreach ($sections as $s) {
                    if (($s->id !== (int)$post['node_id']) && $s->order_by <= $post['position']) {
                        --$s->order_by;
                        $s->save();
                    }
                }
            }
            $updatedSection->order_by = $post['position'];
        }
        if ($updatedSection->save()) {
            return true;
        }
        return false;
    }

    /**
     * @param array $modelArray
     * @param array $constraints
     * @param string $relation
     * @return bool|SingleTreeSection
     */
    public static function findInTree(array $modelArray, array $constraints, string $relation = 'children'): bool|SingleTreeSection
    {
        foreach ($modelArray as $model) {
            $section = $model->findFromConstraints($constraints, $relation);
            if ($section) {
                return $section;
            }
        }
        return false;
    }

    /**
     * @param array $constraints
     * @param string $relation
     * @return bool|SingleTreeSection
     */
    protected function findFromConstraints(array $constraints, string $relation): bool|SingleTreeSection
    {
        if ($this->{$relation}) {
            foreach ($this->{$relation} as $child) {
                $section = $child->findInTree($this->{$relation}, $constraints);
                if ($section) {
                    return $section;
                }
            }
        }
        foreach ($constraints as $field => $constraint) {
            if ($constraint === '') {
                if (!$this->$field) {
                    $section = $this;
                } else {
                    $section = false;
                    break;
                }
            } elseif ($this->$field && $this->$field === $constraint) {
                $section = $this;
            } else {
                $section = false;
                break;
            }
        }
        if (isset($section) && $section) {
            return $section;
        }
        return false;
    }

    /**
     * @param string $type
     * @return bool|SingleTreeSection|MultipleTreeSection
     */
    public function getContentTypeSection(string $type): bool|SingleTreeSection|MultipleTreeSection
    {
        return self::findInTree(Yii::$app->view->params['site-tree'], ['type' => $type]);
    }

    /**
     * @param int $id
     * @return array
     */
    public static function buildSiteTree(mixed $parentQuery = null): array
    {
        if (!$parentQuery) {
            $parentQuery = static::find()->joinWith('defaultTranslation')
                ->select(['id', 'parent_id AS p', 'title AS text', 'slug AS s', 'order_by AS o', 'type AS t'])
                ->where(['parent_id' => NULL])->orderBy(['p' => SORT_ASC, 'o' => SORT_ASC])
                ->asArray();
        }
        $queried = $parentQuery->all();
        $siteTree = [];
        $callback = function ($id, $text, $slug, $orderby, $type, $parent) {
            return [
                'id' => $id,
                'text' => $text,
                'data' => [
                    'p' => $parent,
                    's' => $slug,
                    'o' => $orderby,
                    't' => $type,
                ]
            ];
        };
        static::buildLevel($queried, array_column($queried, 'id'), $siteTree, $callback);
        return $siteTree['firstLevel']['children'] ?? [];
    }

    /**
     * @param array $newValues
     * @param array $ids
     * @param array $siteTree
     * @param callable $callback
     * @return array
     */
    public static function buildLevel(array $newValues, array $ids, array &$siteTree, callable $callback): array
    {
        foreach ($newValues as $section) {
            if ($section['p']) {
                $siteTree[$section['id']] = &$siteTree[$section['p']]['children'][];
            } else {
                $siteTree[$section['id']] = &$siteTree['firstLevel']['children'][];
            }
            $siteTree[$section['id']] = $callback((int)$section['id'], $section['text'], $section['s'], (int)$section['o'], $section['t'], (int)$section['p']);
        }
        $newValues = static::find()->joinWith('defaultTranslation')
            ->select(['id', 'parent_id AS p', 'title AS text', 'slug AS s', 'order_by AS o', 'type AS t'])
            ->andWhere(['in', 'parent_id', $ids])
            ->orderBy(['p' => SORT_ASC, 'o' => SORT_ASC])
            ->asArray()->all();
        if ($newValues) {
            $ids = array_column($newValues, 'id');
            self::buildLevel($newValues, $ids, $siteTree, $callback);
        }
        return $siteTree;
    }

//    /**
//     * @param mixed|null $siteTreeQuery
//     * @return array
//     */
//    public static function buildSiteTree(mixed $siteTreeQuery = null): array
//    {
//        if (!$siteTreeQuery) {
//            $siteTreeQuery = static::find()
//                ->innerJoinWith('defaultTranslation')
//                ->orderBy(['parent_id' => SORT_ASC, 'order_by' => SORT_ASC])
//                ->asArray();
//        }
//        $siteTree = [];
//        foreach ($siteTreeQuery->each() as $section) {
//            static::flattenArray($section);
//            if ($section['parent_id']) {
//                $siteTree[$section['id']] = &$siteTree[$section['parent_id']]['children'][];
//                $section['fullSlug'] = $siteTree[$section['parent_id']]['fullSlug'] . '/' . $section['slug'];
//            } else {
//                $section['fullSlug'] = '/' . Yii::$app->language . '/' . $section['slug'];
//                $siteTree[$section['id']] = &$siteTree['firstLevel']['children'][];
//            }
//            $siteTree[$section['id']] = $section;
//        }
//        return $siteTree['firstLevel']['children'] ?? [];
//    }
//
//    /**
//     * @param array $section
//     */
//    protected static function flattenArray(array &$section): void
//    {
//        foreach ($section['defaultTranslation'] as $key => $value) {
//            $section[$key] = $value;
//        }
//        unset($section['defaultTranslation']);
//    }
}
