<?php

namespace wdigital\cms\siteTree\models;

use wdigital\cms\siteTree\components\ContentTypeClassResolver;
use wdigital\cms\siteTree\interfaces\ContentTypeInterface;
use wdigital\cms\siteTree\Module;
use Yii;
use yii\base\InvalidConfigException;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;

/**
 * This is the model class for table "section".
 *
 * @property int $id
 * @property int $parent_id
 * @property int $language_id
 * @property string $title
 * @property string $slug
 * @property string $type
 * @property string $meta_description
 * @property string $redirect_to
 * @property string $image
 * @property int $order_by
 * @property int $active
 *
 * @property Html[] $htmls
 * @property MultipleTreeSection $parent
 * @property MultipleTreeSection[] $multipleTreeSections
 */
class MultipleTreeSection extends ActiveRecord
{
    public const IMG_DIR = 'uploads/sections';
    public $children;
    public $fullSlug;

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%section}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules(): array
    {
        return [
            [['parent_id', 'language_id', 'order_by', 'active'], 'integer'],
            [['language_id', 'title', 'slug', 'type'], 'required'],
            [['title', 'slug', 'type', 'meta_description', 'redirect_to', 'image'], 'string', 'max' => 255],
            [['language_id'], 'exist', 'skipOnError' => true, 'targetClass' => Yii::$app->getModule('language')->languageClass, 'targetAttribute' => ['language_id' => 'id']],
            [['parent_id'], 'exist', 'skipOnError' => true, 'targetClass' => static::class, 'targetAttribute' => ['parent_id' => 'id']],
        ];
    }

    /**
     * @return array
     */
    public function attributeLabels(): array
    {
        return [
            'parent_id' => Yii::t('site-tree', 'Parent section'),
            'type' => Yii::t('site-tree', 'Section type'),
            'title' => Yii::t('site-tree', 'Title'),
            'slug' => Yii::t('site-tree', 'Slug'),
            'meta_description' => Yii::t('site-tree', 'Meta description'),
            'redirect_to' => Yii::t('site-tree', 'Redirect to'),
            'active' => Yii::t('site-tree', 'Active'),
            'image' => Yii::t('site-tree', 'Image'),
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getParent(): ActiveQuery
    {
        return $this->hasOne(static::class, ['id' => 'parent_id'])->from(['parent' => static::tableName()]);
    }

    /**
     * @return ActiveQuery
     */
    public function getHtml(): ActiveQuery
    {
        return $this->hasMany(Html::class, ['section_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getChildren(): ActiveQuery
    {
        return $this->hasMany(static::class, ['parent_id' => 'id'])->from(['child' => static::tableName()]);
    }

    /**
     * @param array $sections
     * @param null $parentSlug
     * @return array
     */
    public function setChildren(array $sections, $parentSlug = null): array
    {
        $id = $this->id;
        $children = &$this->children;
        if (!$parentSlug) {
            $module = Yii::$app->getModule('language');
            $this->fullSlug = '/' . $module->languageClass::getCodeFromId($this->language_id) . '/' . $this->slug;
        } else {
            $this->fullSlug = $parentSlug . '/' . $this->slug;
        }
        foreach ($sections as $index => $section) {
            if ($section->parent_id === $id) {
                unset($sections[$index]);
                $section->setChildren($sections, $this->fullSlug);
                $children[] = $section;
            }
        }
        return $sections;
    }

    /**
     * @param array $array
     */
    public static function generateSiteTreeStructure(array &$array): void
    {
        foreach ($array as $index => &$section) {
            $section->setChildren($array);
        }
        foreach ($array as $key => $s) {
            if ($s->parent_id !== null) {
                unset($array[$key]);
            }
        }
    }

    /**
     * @return bool
     */
    public function beforeValidate(): bool
    {
        if ((int)$this->parent_id === 0) {
            $this->parent_id = null;
        }
        $this->language_id = Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language);
        return parent::beforeValidate();
    }

    /**
     * @param bool $insert
     * @return bool
     */
    public function beforeSave($insert): bool
    {
        if ($insert) {
            $lastSection = self::find()->orderBy('order_by DESC')->limit(1)->one();
            if ($lastSection) {
                $this->order_by = $lastSection->order_by + 1;
            } else {
                $this->order_by = 1;
            }
        }
        return parent::beforeSave($insert);
    }

    /**
     * @param bool $insert
     * @param array $changedAttributes
     */
    public function afterSave($insert, $changedAttributes)
    {
        if ($this->type === 'html') {
            if (!$insert) {
                $html = Html::find()->where(['section_id' => $this->id])->limit(1)->one();
                if (!$html) {
                    $html = new Html();
                }
            } else {
                $html = new Html();
            }
            $html->section_id = $this->id;
            $html->save();

        }
        return parent::afterSave($insert, $changedAttributes);
    }

    /**
     * @param $section
     * @param $post
     * @return bool|null
     */
    public function changeOrder($section, $post): ?bool
    {
        $model = new self();
        Yii::debug(var_dump($post));
        $updatedSection = $section;
        if ($post['new_parent'] === '#') {
            $updatedSection->parent_id = null;
        } else {
            $updatedSection->parent_id = $post['new_parent'];
        }
        if ($post['new_parent'] === '#') {
            $sections = $model::find()
                ->where(['parent_id' => NULL])
                ->andWhere(['language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language)])
                ->orderBy('order_by ASC')
                ->all();
            if ($post['old_position'] > $post['position']) {
                foreach ($sections as $s) {
                    if ($s->id !== (int) $post['node_id']) {
                        if ($s->order_by >= $post['position']) {
                            ++$s->order_by;
                            $s->save();
                        }
                    }
                }
            } else {
                $count = 0;
                foreach ($sections as $s) {
                    if ($s->id !== (int) $post['node_id']) {
                        if ($count + 1 <= $post['position']) {
                            $s->order_by = $count;
                            $s->save();
                        }
                        ++$count;
                    }
                }
            }
            $updatedSection->order_by = $post['position'];
        } else {
            $sections = $model::find()->where(['parent_id' => $post['new_parent']])
                ->andWhere(['language_id' => Yii::$app->getModule('language')->languageClass::getIdFromCode(Yii::$app->language)])
                ->orderBy('order_by asc')->all();
            if ($post['old_position'] > $post['position']) {
                foreach ($sections as $s) {
                    if ($s->id !== (int) $post['node_id']) {
                        if ($s->order_by >= $post['position']) {
                            ++$s->order_by;
                            $s->save();
                        }
                    }
                }
            } else {
                foreach ($sections as $s) {
                    if ($s->id !== (int) $post['node_id']) {
                        if ($s->order_by <= $post['position']) {
                            --$s->order_by;
                            $s->save();
                        }
                    }
                }
            }
            $updatedSection->order_by = $post['position'];
        }
        if ($updatedSection->save()) {
            return true;
        }
        return false;
    }

    /**
     * @return false|int
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function delete()
    {
        foreach ($this->children as $child) {
            $child->delete();
        }
        $module = Module::getInstance();
        $contentTypes = $module->content_types;
        foreach ($contentTypes as $contentType) {
            if ($contentType['value'] === $this->type) {
                (new $contentType['class']())->delete();
            }
        }
        return parent::delete();
    }

    /**
     * @param array $modelArray
     * @param array $constraints
     * @return bool|MultipleTreeSection
     */
    public static function findInCollection(array $modelArray, array $constraints)
    {
        foreach ($modelArray as $model) {
            $section = $model->findFromConstraints($constraints);
            if ($section) {
                return $section;
            }
        }
        return false;
    }

    /**
     * @param array $constraints
     * @return bool|MultipleTreeSection
     */
    protected function findFromConstraints(array $constraints)
    {
        if ($this->children) {
            foreach ($this->children as $child) {
                $childSection = $child;
                break;
            }
            $section = $childSection->findInCollection($this->children, $constraints);
            if ($section) {
                return $section;
            }
        }
        foreach ($constraints as $field => $constraint) {
            if ($constraint === '') {
                if (!$this->$field) {
                    $section = $this;
                } else {
                    $section = false;
                    break;
                }
            } elseif ($this->$field && $this->$field == $constraint) {
                $section = $this;
            } else {
                $section = false;
                break;
            }
        }
        if ($section) {
            return $section;
        }
        return false;
    }

    /**
     * @param string $type
     * @return bool|MultipleTreeSection
     */
    public function getContentTypeSection(string $type)
    {
        return self::findInCollection(Yii::$app->view->params['site-tree'], ['type' => $type]);
    }

    /**
     * @param array $modelArray
     * @param array $constraints
     * @param string $relation
     * @return bool|SingleTreeSection
     */
    public static function findInTree(array $modelArray, array $constraints, string $relation = 'children')
    {
        foreach ($modelArray as $model) {
            $section = $model->findFromConstraints($constraints, $relation);
            if ($section) {
                return $section;
            }
        }
        return false;
    }

    /**
     * @param string $slugs
     * @return array|bool
     * @throws \yii\base\InvalidConfigException
     */
    public static function renderView(string $slugs)
    {
        $contentTypeSlug = '';
        $section = static::findInTree(\Yii::$app->view->params['site-tree'], ['fullSlug' => $slugs]);
        while (!($section && $section->type)) {
            $slugs = explode('/', $slugs);
            $contentTypeSlug = '/' . array_pop($slugs) . $contentTypeSlug;
            if (!$slugs) {
                break;
            }
            $slugs = implode('/', $slugs);
            $section = static::findInTree(\Yii::$app->view->params['site-tree'], ['fullSlug' => $slugs]);
        }
        if ($section && $section->type) {
            $contentTypeResolver = new ContentTypeClassResolver();
            if ($section->parent_id) {
                $parentSection = static::findInTree(\Yii::$app->view->params['site-tree'], ['id' => $section->parent_id]);
            } else {
                $parentSection = null;
            }
            \Yii::$app->view->params['activeSection'] = $section;
            $classInstance = $contentTypeResolver->getContentTypeClassInstance($section->type);
            if ($classInstance instanceof ContentTypeInterface) {
                $viewContent = $contentTypeSlug ? $classInstance->getDetailedFrontendContent($section, $contentTypeSlug) : $classInstance->getFrontendContent($section);
                $view = $contentTypeSlug ? $classInstance->getContentTypeDetailedView($viewContent) : $classInstance->getContentTypeView();
                return [
                    'view' => $view,
                    'section' => $section,
                    'parentSection' => $parentSection,
                    'content' => $viewContent,
                    'contentTypeSlug' => $contentTypeSlug
                ];
            }
            throw new InvalidConfigException('Specified class instance doesn\'t implement ContentTypeInterface');

        }
        return false;
    }
}
