<?php

namespace wdigital\cms\siteTree\models;

use yii\base\Model;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii2tech\ar\variation\VariationBehavior;

/**
 * This is the model class for table "section".
 *
 * @property int $id
 * @property int $parent_id
 * @property string $type
 * @property int $active
 * @property int $order_by
 * @property string $image
 *
 * @property SingleTreeSectionTranslation[] $sectionTranslations
 */
class SingleTreeSection extends ActiveRecord
{
    public $fullSlug;
    public $children;

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%section}}';
    }

    /**
     * @return array
     */
    public static function fillable(): array
    {
        return [
            'title',
            'slug',
            'redirect_to',
        ];
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            [['parent_id', 'order_by'], 'integer'],
            [['type', 'order_by'], 'required'],
            [['type'], 'string', 'max' => 255],
            [['active'], 'boolean'],
            ['parent_id', 'filter', 'skipOnError' => true, 'filter' => function ($value) {
                if ((int)$value === 0) {
                    return null;
                }
                return $value;
            }],
            [['parent_id'], 'exist', 'skipOnEmpty' => true, 'targetClass' => static::class, 'targetAttribute' => ['parent_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'id' => \Yii::t('site-tree', 'ID'),
            'parent_id' => \Yii::t('site-tree', 'Parent section'),
            'type' => \Yii::t('site-tree', 'Section type'),
            'active' => \Yii::t('site-tree', 'Active section'),
            'order_by' => \Yii::t('site-tree', 'Order By'),
            'image' => \Yii::t('site-tree', 'Image'),
        ];
    }

    /**
     * @inheritdoc
     */
    public function behaviors(): array
    {
        return [
            'translations' => [
                'class' => VariationBehavior::class,
                'variationsRelation' => 'sectionTranslations',
                'defaultVariationRelation' => 'defaultTranslation',
                'variationOptionReferenceAttribute' => 'language_id',
                'optionModelClass' => \Yii::$app->getModule('language')->languageClass,
                'defaultVariationOptionReference' => \Yii::$app->getModule('language')->languageClass::getIdFromCode(\Yii::$app->language),
            ],
        ];
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getSectionTranslations(): \yii\db\ActiveQuery
    {
        return $this->hasMany(SingleTreeSectionTranslation::class, ['section_id' => 'id']);
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getDefaultTranslation(): \yii\db\ActiveQuery
    {
        return $this->hasDefaultVariationRelation();
    }

    /**
     * @param array $array
     */
    public static function generateSiteTreeStructure(array &$array): void
    {
        foreach ($array as $index => &$section) {
            $section->setChildren($array);
        }
        foreach ($array as $key => $s) {
            if ($s->parent_id !== null) {
                unset($array[$key]);
            }
        }
    }

    /**
     * @return \yii\db\ActiveQuery
     */
    public function getChildren(): \yii\db\ActiveQuery
    {
        return $this->hasMany(static::class, ['parent_id' => 'id']);
    }

    /**
     * @param $sections
     * @param null $parentSlug
     * @return mixed
     */
    public function setChildren(&$sections, $parentSlug = null)
    {
        $id = $this->id;
        $children = &$this->children;
        if (!$parentSlug) {
            $module = \Yii::$app->getModule('language');
            $this->fullSlug = '/' . $module->languageClass::getCodeFromId($this->language_id) . '/' . $this->slug;
        } else {
            $this->fullSlug = $parentSlug . '/' . $this->slug;
        }
        foreach ($sections as $index => $section) {
            if ($section->parent_id == $id) {
                unset($sections[$index]);
                $section->setChildren($sections, $this->fullSlug);
                $children[] = $section;
            }
        }
        return $sections;
    }

    /**
     * @param $post
     * @return bool
     */
    public function createNew($post): bool
    {
        $translationModels = ArrayHelper::map(
            $this->getVariationModels(),
            'language_id',
            function ($translatedModel) {
                return $translatedModel;
            }
        );
        if (Model::loadMultiple($translationModels, $post) && $this->setVariationModels($translationModels) && $this->save()) {
            if ($this->type === 'html') {
                $html = new Html([
                    'section_id' => $this->id
                ]);
                $html->getVariationModels();
                return $html->save();
            }
            return true;
        }
        return false;
    }

    /**
     * @param $post
     * @return bool|null
     */
    public function editSection($post): ?bool
    {
        $translationModels = ArrayHelper::map(
            $this->getVariationModels(),
            'language_id',
            function ($translatedModel) {
                return $translatedModel;
            }
        );
        return Model::loadMultiple($translationModels, $post) && $this->setVariationModels($translationModels) && $this->save();
    }

    /**
     * @inheritdoc
     */
    public function beforeValidate(): bool
    {
        if ($this->isNewRecord) {
            $lastSection = self::find()->orderBy('order_by DESC')->limit(1)->one();
            if ($lastSection !== null) {
                $this->order_by = $lastSection->order_by + 1;
            } else {
                $this->order_by = 1;
            }
        }
        return parent::beforeValidate();
    }

    /**
     * @inheritdoc
     */
    public function afterSave($insert, $changedAttributes): void
    {
        if (!$insert && !empty($changedAttributes['type']) && $changedAttributes['type'] == 'html') {
            $html = Html::find()->where(['section_id' => $this->id])->limit(1)->one();
            if (!$html) {
                $html = new Html([
                    'section_id' => $this->id
                ]);
                $html->getVariationModels();
                $html->save(false);
            }
        }
        parent::afterSave($insert, $changedAttributes);
    }

    /**
     * @return bool
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public function beforeDelete(): bool
    {
        $children = $this->getChildren()->orderBy(['order_by' => SORT_ASC])->all();
        if (!empty($children)) {
            foreach ($children as $child) {
                $child->delete();
            }
        }
        $contentTypes = \Yii::$app->getModule('site-tree')->content_types;
        foreach ($contentTypes as $contentType) {
            if ($contentType['value'] === $this->type) {
                (new $contentType['class']())->delete();
            }
        }
        return parent::beforeDelete();
    }
}