<?php

namespace wdigital\cms\i18n\components\translatableModel;

use Yii;
use yii\base\InvalidConfigException;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;

/**
 * @property-read Helper $translatableModelHelper
 * @property string $translatableModelManagerComponent
 * @property string $language
 *
 * @property ActiveRecord[] $translations
 */
trait TranslatableTrait
{
    /**
     * @var string The default name of the manager component for all models of this class.
     */
    protected static $defaultTranslatableModelManagerComponent = 'translatableModelManager';
    protected $_translatableModelManagerComponent;
    protected $_language;

    /**
     * @see parent::__get()
     *
     * @param string $name
     * @return mixed
     */
    public function __get($name)
    {
        $helper = $this->getTranslatableModelHelper();
        if (in_array($name, $helper->attributes)) {
            $language = $this->language;
            if ($language !== Yii::$app->sourceLanguage) {
                $value = $helper->getTranslatedAttribute($this, $name, $language);
                if ($value || !$helper->fallbackToSource) {
                    return $value;
                }
            }
        }

        return parent::__get($name);
    }
    /**
     * @see parent::__set()
     *
     * @param string $name
     * @param mixed $value
     */
    public function __set($name, $value)
    {
        $helper = $this->getTranslatableModelHelper();
        if (in_array($name, $helper->attributes)) {
            $language = $this->language;
            if ($language !== Yii::$app->sourceLanguage) {
                $helper->setTranslatedAttribute($this, $name, $value, $language);
                return;
            }
        }

        parent::__set($name, $value);
    }
    /**
     * @see parent::find()
     *
     * @return ActiveQuery
     */
    public static function find($withoutTranslation = false)
    {

        if($withoutTranslation){
            return parent::find();
        }

        $tableName = static::tableName();

        $languageCodeColumn = 'code';
        $languageCodeAttribute = 'language';
        $languageRelation = $languageCodeAttribute;
        $languageAlias = $languageCodeAttribute;

        $translationRelation = 'translations';
        $translationAlias = 'translation';

        $defaultLanguage = var_export(Yii::$app->sourceLanguage, true);
        $columns = [
            '*',
            $languageCodeAttribute => "($defaultLanguage)",
        ];
        $query = parent::find()
            ->select($columns);

        $translationColumns = [];
        foreach (array_keys(static::getTableSchema()->columns) as $column) {
            $translationColumns[$column] = "{$tableName}.{$column}";
        }
        foreach (static::_getTranslatableModelHelper()->attributes as $attribute) {
            $translationColumns[$attribute] = "{$translationAlias}.{$attribute}";
        }
        $translationColumns[$languageCodeAttribute] = "{$languageAlias}.{$languageCodeColumn}";

        $translationQuery = parent::find()
            ->joinWith([
                "{$translationRelation} {$translationAlias}",
                "{$translationRelation}.{$languageRelation} {$languageAlias}"
                ], false, 'INNER JOIN')
            ->select($translationColumns);

        return parent::find()->from([
                $tableName => $query->union($translationQuery, true),
        ]);
    }
    /**
     * @see parent::hasAttribute()
     *
     * @param string $name
     * @return bool
     */
    public function hasAttribute($name)
    {
        return in_array($name, $this->getTranslatableModelHelper()->attributes) || parent::hasAttribute($name);
    }
    /**
     * @see parent::getAttribute()
     *
     * @param string $name
     * @param string $language
     * @return mixed
     */
    public function getAttribute($name, $language = null)
    {
        $helper = $this->getTranslatableModelHelper();
        if ($language !== Yii::$app->sourceLanguage && in_array($name, $helper->attributes)) {
            return $helper->getTranslatedAttribute($this, $name, $language);
        }

        return parent::getAttribute($name);
    }
    /**
     * @see parent::getAttribute()
     *
     * @param string $name
     * @param mixed $value
     * @param string $language
     */
    public function setAttribute($name, $value, $language = null)
    {
        $helper = $this->getTranslatableModelHelper();
        if ($language !== Yii::$app->sourceLanguage && in_array($name, $helper->attributes)) {
            return $helper->setTranslatedAttribute($this, $name, $value, $language);
        }

        return parent::setAttribute($name, $value);
    }
    /**
     * @see parent::setAttributes()
     *
     * @param array $values
     * @param bool $safeOnly
     */
    public function setAttributes($values, $safeOnly = true)
    {
        if (!empty($values['language'])) {
            $values = ['language' => $values['language']] + $values;
        }
        $translations = ArrayHelper::remove($values, 'translations');
        parent::setAttributes($values, $safeOnly);
        if (!empty($translations)) {
            $this->setTranslations($translations);
        }
    }
    /**
     * @see parent::save()
     *
     * @param bool $runValidation
     * @param bool $attributeNames
     * @return bool
     */
    public function save($runValidation = true, $attributeNames = null)
    {
        if (!parent::save($runValidation, $attributeNames)) {
            return false;
        }
        if ($attributeNames && empty(array_intersect($this->translatableModelHelper->attributes, $attributeNames))) {
            return true;
        }
        return $this->getTranslatableModelHelper()->saveTranslations($this);
    }
    /**
     * @return string
     */
    public function getTranslatableModelManagerComponent()
    {
        return $this->_translatableModelManagerComponent
            ?: static::$defaultTranslatableModelManagerComponent;
    }
    /**
     * @param string $component
     */
    public function setTranslatableModelManagerComponent($component)
    {
        $this->_translatableModelManagerComponent = $component;
    }
    /**
     * @return Helper
     */
    protected function getTranslatableModelHelper()
    {
        return static::_getTranslatableModelHelper($this->getTranslatableModelManagerComponent());
    }
    /**
     * @param string $component
     * @return Helper
     * @throws InvalidConfigException
     */
    protected static function _getTranslatableModelHelper($component = null)
    {
        if (!$component) {
            $component = static::$defaultTranslatableModelManagerComponent;
        }
        if (!Yii::$app->has($component)) {
            throw new InvalidConfigException("The translatable model manager component '{$component}' is not configured properly");
        }
        return Yii::$app->get($component)->get(static::class);
    }
    /**
     * @return string
     */
    public function getLanguage()
    {
        return $this->_language
            ?: Yii::$app->language;
    }
    /**
     * @return bool
     */
    public function hasLanguage()
    {
        return (bool)$this->_language;
    }
    /**
     * @param string $language
     */
    public function setLanguage($language)
    {
        $this->_language = $language;
    }
    /**
     * @return ActiveQuery
     */
    public function getTranslations()
    {
        $helper = $this->translatableModelHelper;
        $indexCallback = function ($row) use ($helper) {
            return $helper->translationIndex($row);
        };

        return $this->hasMany($helper->translationModelClass, $helper->relationKey)
                ->indexBy($indexCallback);
    }
    /**
     * @param ActiveRecord[]|array $translations
     */
    public function setTranslations($translations)
    {
        $this->getTranslatableModelHelper()->setTranslations($this, $translations);
    }
    /**
     * Clears the translation cache of an unsaved instance
     */
    public function __destruct()
    {
        $this->translatableModelHelper->consolidatePersistence($this);
    }
}
