<?php

namespace wdigital\cms\i18n\components\translatableModel;

use wdigital\cms\i18n\models\Language;
use Throwable;
use Yii;
use yii\base\Component;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii\helpers\Inflector;
use yii\helpers\StringHelper;

/**
 * @property array $attributes
 * @property array $relationKey
 * @property string $translationModelClass
 */
class Helper extends Component
{
    /**
     * @var array
     */
    protected $_attributes;
    /**
     * @var array
     */
    protected $_relationKey;
    /**
     * @var string
     */
    protected $_translationModelClass;
    /**
     * @var Cache
     */
    protected $_cache;
    /**
     * @var string
     */
    public $translationLanguageAttribute = 'language_id';
    /**
     * @var bool
     */
    public $fallbackToSource = false;
    /**
     * @var string
     */
    public $cacheClass = Cache::class;
    /**
     * @var array
     */
    public $cacheConfig;
    /**
     * @var string
     */
    public $modelClass;

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();

        $cacheConfig = (array)$this->cacheConfig;
        if (empty($cacheConfig['class'])) {
            $cacheConfig['class'] = $this->cacheClass;
        }
        $cacheConfig['helper'] = $this;
        $this->_cache = Yii::createObject($cacheConfig);
    }
    /**
     * @return array
     */
    public function getAttributes()
    {
        if ($this->_attributes === null) {
            $translated = array_keys($this->modelClass::getTableSchema()->columns);
            $translating = array_keys($this->translationModelClass::getTableSchema()->columns);
            $this->_attributes = array_intersect($translated, $translating);
        }
        return $this->_attributes;
    }
    /**
     * @param array $attributes
     */
    public function setAttributes($attributes)
    {
        $this->_attributes = $attributes;
    }
    /**
     * @return array
     */
    public function getRelationKey()
    {
        if (!$this->_relationKey) {
            $modelName = Inflector::camel2id(StringHelper::basename($this->modelClass), '_');
            $this->_relationKey = ["{$modelName}_id" => 'id'];
        }
        return $this->_relationKey;
    }
    /**
     * @param array $key
     */
    public function setRelationKey($key)
    {
        $this->_relationKey = $key;
    }
    /**
     * @return string
     */
    public function getTranslationModelClass()
    {
        if (!$this->_translationModelClass) {
            $this->_translationModelClass = "{$this->modelClass}Translation";
        }
        return $this->_translationModelClass;
    }
    /**
     * @param string $className
     */
    public function setTranslationModelClass($className)
    {
        $this->_translationModelClass = $className;
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $language
     * @param array $values
     * @return ActiveRecord
     */
    public function createTranslationInstance($model, $language, $values = null)
    {
        if (!$language) {
            $language = $model->language;
        }

        $properties = array_merge((array)$values, [
            $this->translationLanguageAttribute => $this->resolveLanguageID($language),
        ]);
        foreach ($this->relationKey as $source => $related) {
            $properties[$source] = $model->{$related};
        }

        /* @var $instance ActiveRecord */
        $instance = Yii::createObject($this->translationModelClass);
        Yii::configure($instance, $properties);

        return $instance;
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $language
     * @return ActiveRecord
     */
    protected function dbGetTranslation($model, $language = null)
    {
        if (!$language) {
            $language = $model->language;
        }

        $translation = $model->getTranslations()
            ->andWhere([$this->translationLanguageAttribute => $this->resolveLanguageID($language)])
            ->one();

        return $translation;
    }
    /**
     * @param string $code
     * @return int
     */
    protected function resolveLanguageID($code)
    {
        return Language::getIdFromCode($code);
    }
    /**
     * @param array $row
     * @return string
     */
    public function translationIndex($row)
    {
        return Language::getCodeFromId($row[$this->translationLanguageAttribute]);
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $language
     * @return ActiveRecord
     */
    public function getTranslation($model, $language = null)
    {
        $translation = $this->_cache->get($model, $language);

        if (!$translation) {
            $translation = $this->dbGetTranslation($model, $language);
            if (!$translation) {
                $translation = $this->createTranslationInstance($model, $language);
            }
            $this->_cache->set($model, $translation, $language);
        }

        return $translation;
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param ActiveRecord $translation
     * @param string $language
     */
    public function setTranslation($model, $translation, $language = null)
    {
        $values = array_intersect_key(ArrayHelper::toArray($translation), array_fill_keys($this->attributes, null));

        if (!$language) {
            $language = $model->language;
        }

        if ($language === Yii::$app->sourceLanguage) {
            $oldLanguage = $model->hasLanguage()
                ? $model->language
                : null;
            $model->language = $language;
            $model->setAttributes($values, false);
            $model->language = $oldLanguage;
            return;
        }

        $values[$this->translationLanguageAttribute] = $this->resolveLanguageID($language);
        $this->getTranslation($model, $language)
            ->setAttributes($values, false);
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param ActiveRecord[] $translations
     */
    public function getTranslations($model)
    {
        return $this->_cache->getAll($model);
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param ActiveRecord[] $translations
     */
    public function setTranslations($model, $translations)
    {
        foreach ($translations as $language => $translation) {
            $this->setTranslation($model, $translation, $language);
        }
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @return bool
     */
    public function saveTranslations($model)
    {
        $transaction = Yii::$app->db->beginTransaction();
        try {
            foreach ($this->getTranslations($model) as $translation) {
                foreach ($this->relationKey as $source => $related) {
                    $translation->{$source} = $model->{$related};
                }

                $nothingTranslated = true;
                foreach ($this->attributes as $attribute) {
                    if (trim($translation->{$attribute})) {
                        $nothingTranslated = false;
                        break;
                    }
                }
                if ($nothingTranslated) {
                    if (!$translation->isNewRecord) {
                        $translation->delete();
                    }
                    continue; // skip this translation model
                }

                if (!$translation->save()) {
                    $transaction->rollBack();
                    $model->addErrors($translation->errors);
                    return false;
                }
            }
            $transaction->commit();
            return true;
        } catch (Throwable $exception) {
            $transaction->rollBack();
            throw $exception;
        }
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $name
     * @param string $language
     * @return string
     */
    public function getTranslatedAttribute($model, $name, $language = null)
    {
        return $this->getTranslation($model, $language)->{$name};
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $name
     * @param string $value
     * @param string $language
     */
    public function setTranslatedAttribute($model, $name, $value, $language = null)
    {
        $this->getTranslation($model, $language)->{$name} = $value;
    }
    /**
     * Removes non-persisent (unsaved model) translations, ensures persistence of saved ones.
     *
     * @param ActiveRecord|TranslatableTrait $model
     */
    public function consolidatePersistence($model)
    {
        $this->_cache->consolidatePersistence($model);
    }
}
