<?php

namespace wdigital\cms\i18n\components\translatableModel;

use yii\base\Component;
use yii\db\ActiveRecord;

/**
 */
class Cache extends Component
{
    /**
     * @var array
     */
    protected $_data = [];
    /**
     * @var Helper
     */
    public $helper;

    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @return string
     */
    protected function getKey($model)
    {
        $key = [];
        foreach ($this->helper->relationKey as $related) {
            $key[$related] = $model->{$related};
        }
        return json_encode($key);
    }
    /**
     * @param string $key
     * @return bool
     */
    protected function isKeyValid($key)
    {
        return !empty(array_filter(json_decode($key, true)));
    }
    /**
     * @param string $key
     * @param string $hash
     * @return bool
     */
    protected function isNotPersisted($key, $hash)
    {
        return empty($this->_data[$key]) && !empty($this->_data[$hash]);
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @return string
     */
    protected function generateKey($model)
    {
        $key = $this->getKey($model);
        $hash = spl_object_hash($model);

        if (!$this->isKeyValid($key)) {
            return $hash;
        }

        if ($this->isNotPersisted($key, $hash)) {
            $this->rename($hash, $key);
        }

        return $key;
    }
    /**
     * @param string $oldKey
     * @param string $newKey
     */
    protected function rename($oldKey, $newKey)
    {
        $this->_data[$newKey] = $this->_data[$oldKey];
        unset($this->_data[$oldKey]);
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param ActiveRecord[] $translations
     */
    public function setAll($model, $translations)
    {
        foreach ($translations as $language => $translation) {
            $this->set($model, $translation, $language);
        }
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param ActiveRecord $translation
     * @param string $language
     */
    public function set($model, $translation, $language = null)
    {
        if (!$language) {
            $language = $model->language;
        }
        $cacheKey = $this->generateKey($model);
        if (empty($this->_data[$cacheKey])) {
            $this->_data[$cacheKey] = [];
        }
        $this->_data[$cacheKey][$language] = $translation;
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @return ActiveRecord[]
     */
    public function getAll($model)
    {
        $cacheKey = $this->generateKey($model);
        if (empty($this->_data[$cacheKey])) {
            $this->_data[$cacheKey] = [];
        }
        return $this->_data[$cacheKey];
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $language
     * @return ActiveRecord
     */
    public function get($model, $language = null)
    {
        if (!$language) {
            $language = $model->language;
        }
        $translations = $this->getAll($model);
        return $translations[$language]
            ?? null;
    }
    /**
     * @param ActiveRecord|TranslatableTrait $model
     * @param string $language
     */
    public function remove($model, $language = null)
    {
        $cacheKey = $this->generateKey($model);
        if (empty($this->_data[$cacheKey])) {
            return;
        }
        if (!$language) {
            $language = $model->language;
        }
        unset($this->_data[$cacheKey][$language]);
    }
    /**
     * Removes non-persisent (unsaved model) translations, ensures persistence of saved ones.
     *
     * @param ActiveRecord|TranslatableTrait $model
     */
    public function consolidatePersistence($model)
    {
        $key = $this->getKey($model);
        $hash = spl_object_hash($model);

        if (!$this->isKeyValid($key)) {
            if (isset($this->_data[$hash])) {
                unset($this->_data[$hash]);
            }
            return;
        }

        if ($this->isNotPersisted($key, $hash)) {
            $this->rename($hash, $key);
        }
    }
}
