<?php

namespace wdigital\cms\gallery\controllers;

use Throwable;
use wdigital\cms\gallery\models\multi\Photo;
use Yii;
use yii\db\Exception;
use yii\db\StaleObjectException;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\web\Response;
use yii\web\UnprocessableEntityHttpException;

/**
 * PhotoController implements the CRUD actions for Photo model.
 */
class PhotoController extends Controller
{

    /**
     * Updates an existing Photo model.
     * If update is successful, the browser will be redirected to either 'gallery page' or 'gallery index' page.
     * @param integer $id
     * @return Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate(int $id): Response
    {
        $model = $this->findModel($id);
        $gallery = $model->gallery;
        if ($model->load(Yii::$app->request->post()) && $model->save() && $gallery !== null) {
            return $this->redirect(['default/update', 'id' => $gallery->id]);
        }
        if ($gallery !== null) {
            return $this->redirect(['default/update', 'id' => $gallery->id]);
        }
        return $this->redirect(['default/index']);
    }

    /**
     * Updates an existing Photo models description.
     * If update is successful, a successful JSON message will be sent.
     * @param integer $id
     * @throws UnprocessableEntityHttpException if model could not be updated
     * @throws NotFoundHttpException if the request is not of Ajax nature
     */
    public function actionUpdatePhotoDescription(int $id)
    {
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            $model = $this->findModel($id);
            if ($model->load(Yii::$app->request->post()) && $model->save()) {
                Yii::$app->response->data = ['data' => ['message' => Yii::t('gallery', 'Gallery photo modified')]];
                return;
            }
            throw new UnprocessableEntityHttpException(Yii::t('gallery', 'Gallery photo could not be modified'));
        }
        throw new NotFoundHttpException();
    }

    /**
     * Updates multiple existing Photo models order.
     * If update is successful, a successful JSON message will be sent.
     * @throws NotFoundHttpException if the request is not of Ajax nature
     */
    public function actionUpdatePhotoOrder()
    {
        if(Yii::$app->request->isAjax){
            Yii::$app->response->format = Response::FORMAT_JSON;
            $post = Yii::$app->request->post();
            $query = Photo::find()->where(['in', 'id', array_keys($post)]);
            foreach ($query->each() as $photo) {
                /** @var Photo $photo */
                if (isset($post[$photo->id])) {
                    $photo->order_by = $post[$photo->id];
                    $photo->save();
                }
            }
            Yii::$app->response->data = ['data' => ['message' => Yii::t('gallery', 'Gallery photos modified')]];
            return;
        }

        throw new NotFoundHttpException();
    }

    /**
     * Deletes an existing Photo model.
     * If deletion is successful, the browser will be redirected to either 'gallery page' or 'gallery index' page.
     * @param integer $id
     * @return mixed
     * @throws NotFoundHttpException
     * @throws Throwable
     * @throws StaleObjectException
     */
    public function actionDelete($id, $resourceId)
    {
        $model = $this->findModel($resourceId);
        $gallery = $model->gallery;
        $model->destroy();
        if($gallery !== null){
            return $this->redirect(['default/update', 'id' => $id, 'resourceId' => $gallery->id]);
        }
        return $this->redirect(['default/index', 'id' => $id]);

    }

    /**
     * Finds the Photo model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Photo the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id): Photo
    {
        $photoClass = Yii::$app->getModule('gallery')->photoClass;
        if (($model = $photoClass::find()->where([$photoClass::tableName().'.id' => $id])->limit(1)->one()) !== null) {
            return $model;
        }
        throw new NotFoundHttpException();
    }


    /**
     * Rotates an existing Photo file.
     * If rotation is successful, a successful JSON message will be sent.
     * @param integer $id
     * @throws NotFoundHttpException if the request is not of Ajax nature
     * @throws UnprocessableEntityHttpException if model could not be rotated
     * @throws \yii\base\Exception
     */
    public function actionRotatePhoto(int $id)
    {
        if (Yii::$app->request->isAjax) {
            Yii::$app->response->format = Response::FORMAT_JSON;
            $model = $this->findModel($id);
            if ($model->rotatePhoto(Yii::$app->request->post())) {
                Yii::$app->response->data = ['data' => ['message' => Yii::t('gallery', 'Gallery photo rotated')]];
                return;
            }
            throw new UnprocessableEntityHttpException(Yii::t('gallery', 'Gallery photo could not be modified'));
        }
        throw new NotFoundHttpException();
    }

}
