<?php

namespace wdigital\cms\gallery\models\single;

use DateTime;
use wdigital\cms\gallery\components\FileControl;
use Yii;
use yii\base\InvalidConfigException;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii\web\UploadedFile;

/**
 * This is the model class for table "{{%gallery}}".
 *
 * @property integer $id
 * @property string $title
 * @property string $description
 * @property string $date
 * @property string $author
 * @property integer $published
 *
 * @property GalleryType[] $galleryTypes
 * @property Photo[] $photos
 */
class Gallery extends ActiveRecord
{

    /**
     * @var
     */
    private $_galleryTypes;

    /**
     * Publically accessable subdirectory for file storage.
     */
    const IMG_DIR = 'gallery/photos';

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return '{{%gallery}}';
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            ['title', 'required'],
            [['description'], 'string'],
            [['date', 'galleryTypes'], 'safe'],
            [['date'], 'date', 'format' => 'php:d.m.Y'],
            ['date', 'filter', 'filter' => function ($value) {
                    if (!empty($value)) {
                        $date = DateTime::createFromFormat('d.m.Y', $value);
                        if ($date) {
                            return $date->format('Y-m-d');
                        }
                        return date('Y-m-d');
                    }
                return date('Y-m-d');
            }],
            [['published'], 'integer'],
            [['title', 'author'], 'string', 'max' => 255],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'id' => Yii::t('gallery', 'ID'),
            'title' => Yii::t('gallery', 'Title'),
            'description' => Yii::t('gallery', 'Description'),
            'date' => Yii::t('gallery', 'Date'),
            'author' => Yii::t('gallery', 'Author'),
            'published' => Yii::t('gallery', 'Published'),
            'galleryTypes' => Yii::t('gallery', 'Related gallery types'),
        ];
    }

    /**
     * @return ActiveQuery
     * @throws InvalidConfigException
     */
    public function getGalleryTypes(): ActiveQuery
    {
        return $this->hasMany(GalleryType::class, ['id' => 'gallery_type_id'])->viaTable('{{%gallery_to_gallery_type}}', ['gallery_id' => 'id']);
    }

    /**
     * @param array $galleryTypes
     * @return void
     */
    public function setGalleryTypes(array $galleryTypes)
    {
        if(empty($galleryTypes)){
            $galleryTypes = [];
        }
        $this->_galleryTypes = $galleryTypes;

        $saveGalleryTypes = function (){
            $this->unlinkAll('galleryTypes', true);
            $relatedClass = Yii::$app->getModule('gallery')->galleryTypeClass;
            $relatedClass = new $relatedClass;
            $query = $relatedClass::find()->where(['in', 'id', $this->_galleryTypes]);
            if(Yii::$app->getModule('gallery')->translate
                && ArrayHelper::keyExists('wdigital\cms\i18n\components\translatableModel\TranslatableTrait', class_uses($relatedClass))) {
                $query->andWhere([$relatedClass::tableName().'.language' => Yii::$app->language]);
            }
            $relatedClassInstances = $query->all();
            foreach ($relatedClassInstances as $relatedClassInstance){
                $this->link('galleryTypes', $relatedClassInstance);
            }
        };

        $this->on(self::EVENT_AFTER_INSERT, $saveGalleryTypes);
        $this->on(self::EVENT_AFTER_UPDATE, $saveGalleryTypes);
    }


    /**
     * @return ActiveQuery
     */
    public function getPhotos(): ActiveQuery
    {
        return $this->hasMany(Photo::class, ['gallery_id' => 'id']);
    }

    /**
     * Creates related gallery photo instances and saves its files on filesystem.
     * Sent files need to have a name of images as input field.
     * @var PhotoUpload $photoUpload
     * @return bool
     */
    public function saveRelatedPhotos(PhotoUpload $photoUpload): bool
    {
        $photoUpload->images = UploadedFile::getInstances($photoUpload, 'images');
        if ($photoUpload->validate()) {
            foreach ($photoUpload->images as $post) {
                $photo = new Photo();
                if ($photo->create($post, $this)) {
                    FileControl::uploadFile($post, self::IMG_DIR, $photo->filename);
                }
            }
            return true;
        }
        return false;
    }
}
