<?php

namespace wdigital\cms\gallery\models\multi;


use DateTime;
use Exception;
use wdigital\cms\gallery\components\FileControl;
use wdigital\cms\gallery\models\single\PhotoUpload;
use wdigital\cms\i18new\components\ActiveRecordHelper;
use wdigital\cms\i18new\models\Language;
use Yii;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\behaviors\TimestampBehavior;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\ArrayHelper;
use yii\web\UploadedFile;
use yii2tech\ar\variation\VariationBehavior;

/**
 * This is the model class for table "gallery".
 *
 * @property int $id
 * @property string $date
 * @property string $created_at
 * @property string $updated_at
 * @property string $author
 * @property int $published
 *
 * @property GalleryType[] $galleryTypes
 * @property GalleryTranslation[] $galleryTranslations
 * @property Language[] $languages
 * @property Photo[] $photos
 * @method hasDefaultVariationRelation()
 * @method getVariationModels()
 * @method setVariationModels(array $translationModels)
 */
class Gallery extends ActiveRecord
{
    /**
     * @var array
     */
    private $_galleryTypes;

    /**
     * Publically accessable subdirectory for file storage.
     */
    const IMG_DIR = 'gallery/photos';

    /**
     * @return string
     */
    public static function primaryField(): string
    {
        return 'title';
    }

    /**
     * @return array
     * @throws Exception
     */
    public function behaviors(): array
    {
        return [
            'translations' => [
                'class' => VariationBehavior::class,
                'variationsRelation' => 'galleryTranslations',
                'defaultVariationRelation' => 'defaultTranslation',
                'variationOptionReferenceAttribute' => 'language_id',
                'optionModelClass' => Language::class,
                'defaultVariationOptionReference' => Language::getIdFromCode(Yii::$app->language),
                'variationSaveFilter' => [ActiveRecordHelper::class, 'filterVariationSave'],
            ],
            [
                'class' => TimestampBehavior::class,
                'value' => (new DateTime())->format('Y-m-d H:i:s'),
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return 'gallery';
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            [['date', 'created_at', 'updated_at',  'galleryTypes'], 'safe'],
            [['date'], 'date', 'format' => 'php:d.m.Y'],
            ['date', 'filter', 'filter' => function ($value) {
                if (!empty($value)) {
                    $date = DateTime::createFromFormat('d.m.Y', $value);
                    if ($date) {
                        return $date->format('Y-m-d');
                    }
                    return date('Y-m-d');
                }
                return date('Y-m-d');
            }],
            [['author'], 'string', 'max' => 255],
            [['published'], 'integer'],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'id' => 'ID',
            'date' => 'Date',
            'created_at' => 'Created At',
            'updated_at' => 'Updated At',
            'author' => 'Author',
            'published' => 'Published',
        ];
    }

    /**
     * @return ActiveQuery
     * @throws InvalidConfigException
     */
    public function getGalleryTypes(): ActiveQuery
    {
        return $this->hasMany(GalleryType::class, ['id' => 'gallery_type_id'])->viaTable('gallery_to_gallery_type', ['gallery_id' => 'id']);
    }

    /**
     * @param $galleryTypes
     * @return void
     */
    public function setGalleryTypes($galleryTypes)
    {
        if (empty($galleryTypes)) {
            $galleryTypes = [];
        }
        $this->_galleryTypes = $galleryTypes;

        $saveGalleryTypes = function () {
            $this->unlinkAll('galleryTypes', true);
            $relatedClass = Yii::$app->getModule('gallery')->galleryTypeClass;
            $relatedClass = new $relatedClass;
            $query = $relatedClass::find()->where(['in', 'id', $this->_galleryTypes]);
            $relatedClassInstances = $query->all();
            foreach ($relatedClassInstances as $relatedClassInstance) {
                $this->link('galleryTypes', $relatedClassInstance);
            }
        };

        $this->on(self::EVENT_AFTER_INSERT, $saveGalleryTypes);
        $this->on(self::EVENT_AFTER_UPDATE, $saveGalleryTypes);
    }

    /**
     * @return ActiveQuery
     */
    public function getGalleryTranslations(): ActiveQuery
    {
        return $this->hasMany(GalleryTranslation::class, ['gallery_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getPhotos(): ActiveQuery
    {
        return $this->hasMany(Photo::class, ['gallery_id' => 'id']);
    }

    /**
     * Creates related gallery photo instances and saves its files on filesystem.
     * Sent files need to have a name of images as input field.
     * @var PhotoUpload $photoUpload
     * @return bool
     */
    public function saveRelatedPhotos(PhotoUpload $photoUpload): bool
    {
        $photoUpload->images = UploadedFile::getInstances($photoUpload, 'images');
        if ($photoUpload->validate()) {
            foreach ($photoUpload->images as $post) {
                $photo = new Photo();
                if ($photo->create($post, $this)) {
                    FileControl::uploadFile($post, self::IMG_DIR, $photo->filename);
                }
            }
            return true;
        }
        return false;
    }

    /**
     * @return ActiveQuery
     */
    public function getDefaultTranslation(): ActiveQuery
    {
        return $this->hasDefaultVariationRelation();
    }

    /**
     * @param array $data
     * @param null $formName
     * @return bool
     */
    public function load($data, $formName = null): bool
    {
        $translationModels = ArrayHelper::map(
            $this->getVariationModels(),
            'language_id',
            function ($translatedModel) {
                return $translatedModel;
            }
        );
        return parent::load($data, $formName) && Model::loadMultiple($translationModels, $data) && $this->setVariationModels($translationModels);
    }

}