<?php

namespace wdigital\cms\gallery\models\multi;


use Throwable;
use wdigital\cms\gallery\components\FileControl;
use wdigital\cms\i18new\components\ActiveRecordHelper;
use wdigital\cms\i18new\models\Language;
use Yii;
use yii\base\Model;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\db\StaleObjectException;
use yii\helpers\ArrayHelper;
use yii\web\UploadedFile;
use yii2tech\ar\variation\VariationBehavior;

/**
 * This is the model class for table "photo".
 *
 * @property int $id
 * @property int $gallery_id
 * @property string $filename
 * @property int $order_by
 *
 * @property Gallery $gallery
 * @property PhotoTranslation[] $photoTranslations
 * @property Language[] $languages
 * @method hasDefaultVariationRelation()
 * @method getVariationModels()
 * @method setVariationModels(array $translationModels)
 */
class Photo extends ActiveRecord
{
    /**
     * Publicly accessible subdirectory for file storage.
     */
    const IMG_DIR = 'gallery/photos';

    /**
     * @return string
     */
    public static function primaryField(): string
    {
        return 'filename';
    }

    /**
     * @return array
     */
    public function behaviors(): array
    {
        return [
            'translations' => [
                'class' => VariationBehavior::class,
                'variationsRelation' => 'photoTranslations',
                'defaultVariationRelation' => 'defaultTranslation',
                'variationOptionReferenceAttribute' => 'language_id',
                'optionModelClass' => Language::class,
                'defaultVariationOptionReference' => Language::getIdFromCode(Yii::$app->language),
                'variationSaveFilter' => [ActiveRecordHelper::class, 'filterVariationSave'],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public static function tableName(): string
    {
        return 'photo';
    }

    /**
     * @inheritdoc
     */
    public function rules(): array
    {
        return [
            [['gallery_id'], 'required'],
            [['gallery_id', 'order_by'], 'integer'],
            [['filename'], 'string', 'max' => 255],
            [['gallery_id'], 'exist', 'skipOnError' => true, 'targetClass' => Gallery::class, 'targetAttribute' => ['gallery_id' => 'id']],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels(): array
    {
        return [
            'id' => 'ID',
            'gallery_id' => 'Gallery ID',
            'filename' => 'Filename',
            'order_by' => 'Order By',
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getGallery(): ActiveQuery
    {
        return $this->hasOne(Gallery::class, ['id' => 'gallery_id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getPhotoTranslations(): ActiveQuery
    {
        return $this->hasMany(PhotoTranslation::class, ['photo_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getDefaultTranslation(): ActiveQuery
    {
        return $this->hasDefaultVariationRelation();
    }

    /**
     * @param array $data
     * @param null $formName
     * @return bool
     */
    public function load($data, $formName = null): bool
    {
        $translationModels = ArrayHelper::map(
            $this->getVariationModels(),
            'language_id',
            function ($translatedModel) {
                return $translatedModel;
            }
        );
        return parent::load($data, $formName) && Model::loadMultiple($translationModels, $data) && $this->setVariationModels($translationModels);
    }

    /**
     * Creates a new Photo model and generates its filename.
     * @param UploadedFile $data UploadedFile instance that contains file.
     * @param Gallery $model Gallery instance that newly create model will be related to.
     * @return bool
     */
    public function create(UploadedFile $data, Gallery $model): bool
    {
        $this->gallery_id = $model->id;
        $this->filename = FileControl::generateFileName($data);
        $last_photo = self::find()->where(['gallery_id' => $model->id])->orderBy('order_by desc')->limit(1)->one();
        if (!$last_photo) {
            $this->order_by = 0;
        } else {
            $this->order_by = $last_photo->order_by + 1;
        }
        return $this->save();
    }

    /**
     * Deletes an existing Photo model and its related file on filesystem.
     * @return bool
     * @throws Throwable
     * @throws StaleObjectException
     */
    public function destroy() :bool
    {
        if (FileControl::deleteFile(Gallery::IMG_DIR, $this->filename)) {
            return !($this->delete() === false);
        }
        return false;
    }

    /**
     * Returns models file location as URL.
     * @return string
     */
    public function getImageUrl(): string
    {
        return '/' .self::IMG_DIR."/{$this->filename}";
    }


    /**
     * Rotates an existing photo file
     * @param $data
     * @return bool
     * @throws \yii\base\Exception
     */
    public function rotatePhoto($data): bool
    {
        $pathinfo = pathinfo(Yii::getAlias('@app/web') . '/' . self::IMG_DIR . '/' . $this->filename);
        $source = imagecreatefromjpeg(Yii::getAlias('@app/web') . '/' . self::IMG_DIR . '/' . $this->filename);
        unlink(Yii::getAlias('@app/web') . '/' . self::IMG_DIR . '/' . $this->filename);
        $rotate = imagerotate($source, $data['direction'], 0);
        $newFile = Yii::$app->security->generateRandomString() . '.' . $pathinfo['extension'];
        $this->setAttribute('filename', $newFile);
        return ($this->save() && imagejpeg($rotate, Yii::getAlias('@app/web') . '/' . self::IMG_DIR . '/' . $newFile));
    }


}