<?php

namespace wdigital\cms\gallery\components;

use Yii;
use yii\base\Component;
use yii\web\Response;
use yii\web\ResponseFormatterInterface;
use yii\web\JsonResponseFormatter;
use yii\helpers\ArrayHelper;

/**
 * JSend-compliant response formatter.
 */
class JsendResponseFormatter extends Component implements ResponseFormatterInterface
{
    /**
     * @var ResponseFormatterInterface
     */
    protected $jsonFormatterInstance;
    /**
     * @var string Fully qualified name of the class to use for formatting JSON.
     */
    public $jsonFormatterClass = JsonResponseFormatter::class;
    /**
     * @var array Configuration for the JSON formatter.
     */
    public $jsonFormatterConfig = [];
    /**
     * @var string Key for the entry in Response's data holding the error message.
     */
    public $errorMessageKey = 'message';
    /**
     * @var string Key for the entry in Response's data holding the error code.
     */
    public $errorCodeKey = 'code';
    /**
     * @var string Key for the entry in Response's data holding the redirection URL.
     */
    public $redirectUrlKey = 'url';

    /**
     * @inheritdoc
     */
    public function init()
    {
        if (empty($this->jsonFormatterConfig['class'])) {
            $this->jsonFormatterConfig['class'] = $this->jsonFormatterClass;
        }
        $this->jsonFormatterInstance = Yii::createObject($this->jsonFormatterConfig);
    }
    /**
     * Formats the specified response.
     * @param Response $response the response to be formatted.
     */
    public function format($response)
    {
        if ($response->isClientError) {
            $response->data = static::fail($response->data, $response->statusCode);
        } else if ($response->isServerError) {
            $data = $response->data;

            $message = ArrayHelper::remove($data, $this->errorMessageKey, $response->statusText);
            $code = ArrayHelper::remove($data, $this->errorCodeKey, $response->statusCode);

            $response->data = static::error($message, $code, $data);
        } else if ($response->isRedirection) {
            $response->data[$this->redirectUrlKey] = $response->headers->remove('Location');
        } else {
            $response->data = static::success($response->data);
        }
        $this->jsonFormatterInstance->format($response);
    }
    /**
     * @param mixed $data
     * @return array
     */
    public static function success($data = null)
    {
        return [
            'status' => 'success',
            'data' => $data,
        ];
    }
    /**
     * When an API call fails due to an error on the server (HTTP 5xx series).
     *
     * @param string $message
     * @param int $code
     * @param mixed $data
     * @return array
     */
    public static function error($message, $code = null, $data = null)
    {
        $response = [
            'status' => 'error',
            'message' => $message,
        ];
        if ($code !== null) {
            $response['code'] = $code;
        }
        if ($data !== null) {
            $response['data'] = $data;
        }
        return $response;
    }
    /**
     * When an API call is rejected due to invalid data or call conditions (HTTP 4xx series).
     *
     * @param mixed $data
     * @param int $code
     * @return array
     */
    public static function fail($data, $code = null)
    {
        $response = [
            'status' => 'fail',
            'data' => $data,
        ];
        if ($code !== null) {
            $response['code'] = $code;
        } else if(!empty($data['code'])) {
            $response['code'] = $data['code'];
        }
        return $response;
    }
}
