<?php

namespace wdigital\cms\gallery\models;

use DateTime;
use wdigital\cms\gallery\components\FileControl;
use Yii;
use yii\db\ActiveQuery;
use yii\db\ActiveRecord;
use yii\helpers\StringHelper;
use yii\web\UploadedFile;

/**
 * This is the model class for table "{{%gallery}}".
 *
 * @property integer $id
 * @property string $title
 * @property string $description
 * @property string $date
 * @property string $author
 * @property integer $published
 *
 * @property GalleryType[] $galleryTypes
 * @property Photo[] $photos
 */
class Gallery extends ActiveRecord
{
    const IMG_DIR = 'gallery/photos';
    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return '{{%gallery}}';
    }

    /**
     * @inheritdoc
     */
    public function rules()
    {
        return [
            ['title', 'required'],
            [['description'], 'string'],
            [['date'], 'safe'],
            [['date'], 'date', 'format' => 'php:d.m.Y'],
            ['date', 'filter', 'filter' => function ($value) {
                    if (!empty($value)) {
                        $date = DateTime::createFromFormat('d.m.Y', $value);
                        if ($date) {
                            return $date->format('Y-m-d');
                        } else {
                            return date('Y-m-d');
                        }
                    } else {
                        return date('Y-m-d');
                    }
            }],
            [['published'], 'integer'],
            [['title', 'author'], 'string', 'max' => 255],
        ];
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'id' => Yii::t('gallery', 'ID'),
            'title' => Yii::t('gallery', 'Title'),
            'description' => Yii::t('gallery', 'Description'),
            'date' => Yii::t('gallery', 'Date'),
            'author' => Yii::t('gallery', 'Author'),
            'published' => Yii::t('gallery', 'Published'),
        ];
    }

    /**
     * @return ActiveQuery
     */
    public function getGalleryTypes()
    {
        return $this->hasMany(GalleryType::className(), ['id' => 'gallery_type_id'])->viaTable('{{%gallery_to_gallery_type}}', ['gallery_id' => 'id']);
    }

    /**
     * @return ActiveQuery
     */
    public function getPhotos()
    {
        return $this->hasMany(Photo::className(), ['gallery_id' => 'id']);
    }

    /**
     * Links many to many relations based on POST data.
     * Class that extends ActiveRecord needs to have predefined many to many relations;
     * Previously related records will be unlinked, and new ones will be added.
     * @return void
     */
    public function linkRelated()
    {
        $shortName = StringHelper::basename(get_class($this));
        $modelPost = Yii::$app->request->post($shortName);
        if (is_array($modelPost)) {
            foreach ($modelPost as $postKey => $postValue) {
                if ($this->getRelation($postKey, false)) {
                    $this->unlinkAll($postKey, true);
                    if (!empty($modelPost[$postKey])) {
                        $ids = $modelPost[$postKey];
                        $relClass = $this->getRelation($postKey, false)->modelClass;
                        $relArray = $relClass::find()->where(['in', 'id', $ids])->all();
                        foreach ($relArray as $relModel) {
                            $this->link($postKey, $relModel);
                        }
                    }
                }
            }
        }
        return;
    }

    /**
     * Creates related gallery photo instances and saves its files on filesystem.
     * Sent files need to have a name of images as input field.
     * @var PhotoUpload $photoUpload
     * @return bool
     */
    public function saveRelatedPhotos(PhotoUpload $photoUpload)
    {
        $photoUpload->images = UploadedFile::getInstances($photoUpload, 'images');
        if ($photoUpload->validate()) {
            foreach ($photoUpload->images as $post) {
                $photo = new Photo();
                if ($photo->create($post, $this)) {
                    FileControl::uploadFile($post, self::IMG_DIR, $photo->filename);
                }
            }
            return true;
        } else {
            return false;
        }
    }
}
