<?php

namespace wdigital\cms\dynamicContentModel\models;

use wdigital\cms\dynamicContentModel\components\HasImageTrait;
use Throwable;
use Yii;
use yii\base\Exception;
use yii\base\InvalidConfigException;
use yii\base\Model;
use yii\db\ActiveQuery;
use yii\web\NotFoundHttpException;

abstract class AbstractContentModel extends Model
{
    use HasImageTrait;

    public function imageAttributes(): array
    {
        return [];
    }

    /**
     * @param int|null $id
     * @return static
     * @throws InvalidConfigException
     */
    public static function instantiate(int $id = null): static
    {
        $content = self::initBaseModel($id);
        $model = (new static());
        $model->setAttributes($content->value);
        return $model;
    }

    /**
     * @throws Exception
     */
    public function insertOrUpdate(int $id = null): bool
    {
        $content = self::initBaseModel($id);
        $this->processImages();
        $content->value = $this->getAttributes();
        return $content->save();
    }

    /**
     * @throws InvalidConfigException
     */
    private static function initBaseModel(int $id = null): Content
    {
        $classKey = static::class;
        $where = [
            'key' => $classKey,
            'content.id' => $id,
        ];
        /* @var $contentModel Content */
        $contentModel = Yii::createObject(Content::class);
        $content = $contentModel::find()
            ->innerJoinWith('defaultTranslation')
            ->where($where)
            ->limit(1)
            ->one();
        if (!$content) {
            $content = new $contentModel(['key' => $classKey]);
        }
        return $content;
    }

    /**
     * @throws Exception
     */
    public function processImages(): void
    {
        $attributes = $this->imageAttributes();
        foreach ($attributes as $attribute) {
            $this->processImage($attribute);
        }
    }

    /**
     * @throws InvalidConfigException
     */
    public static function findAll(): ActiveQuery
    {
        $classKey = static::class;
        /* @var $contentModel Setting */
        $contentModel = Yii::createObject(Content::class);
        return $contentModel::find()
            ->innerJoinWith('defaultTranslation')
            ->where(['key' => $classKey]);
    }

    /**
     * @throws InvalidConfigException
     */
    public static function all(): array
    {
        $classKey = static::class;
        /* @var $contentModel Setting */
        $contentModel = Yii::createObject(Content::class);
        $contents = $contentModel::find()
            ->innerJoinWith('defaultTranslation')
            ->where(['key' => $classKey])
            ->orderBy('position')
            ->all();
        $models = [];
        foreach ($contents as $content) {
            $model = (new static($content->value));
            $models[] = $model;
        }
        return $models;
    }

    /**
     * @throws Throwable
     * @throws NotFoundHttpException
     */
    public static function delete(int $id): void
    {
        /* @var $contentModel Setting */
        $contentModel = Yii::createObject(Content::class);
        $content = $contentModel::findOne($id);
        if ($content === null) {
            throw new NotFoundHttpException();
        }
        $imageAttributes = (new static)->imageAttributes();
        $model = (new static($content->value));
        foreach ($imageAttributes as $imageAttribute) {
            $model->deleteImage($imageAttribute);
        }
        $content->delete();
    }
}
