<?php

namespace wdigital\cms\audit;

use wdigital\cms\audit\models\Audit;
use Yii;
use yii\base\BootstrapInterface;
use yii\base\InvalidArgumentException;
use yii\base\InvalidConfigException;
use yii\base\Module as BaseModule;
use yii\console\Application as ConsoleApplication;
use yii\helpers\ArrayHelper;
use yii\i18n\PhpMessageSource;
use yii\web\Application as WebApplication;
use yii\web\GroupUrlRule;

class Module extends BaseModule implements BootstrapInterface
{
    /**
     * Values to be redacted out of the audit data (not stored).
     * @var array
     */
    public $redactedValues = [
        'POST.password' => '<password redacted>',
        'POST.User.password' => '<password redacted>',
        'POST.login-form.password' => '<password redacted>',
    ];

    /**
     * The prefix for the module URLs.
     * @see [[GroupUrlRule::prefix]]
     * @var string
     */
    public $urlPrefix = 'audit';

    /**
     * The rules to be used in URL management.
     * @var array
     */
    public $urlRules = [
        '' => 'default/index',
        '<action:\w+>' => 'default/<action>',
    ];

    /**
     * Console application command (controller ID) for running migrations.
     * @var string
     */
    public $migrateCommand = 'migrate';

    /**
     * @inheritdoc
     * @param WebApplication|ConsoleApplication $app
     * @throws InvalidConfigException
     */
    public function bootstrap($app)
    {
        if ($app instanceof WebApplication) {
            $this->addUrlRules($app);
        }
    }

    /**
     * Add an audit entry. Data is automatically redacted according to `$this->redactedValues`.
     *
     * @param string $category
     * @param string $message
     * @param array $data
     * @param int $user_id
     * @throws InvalidConfigException
     */
    public function log($category, $message = null, $data = null, $user_id = null, $frontend_user = null)
    {

        if (!$user_id) {
            $user_id = Yii::$app->user->id;
        }
        if($frontend_user){
            $user_id = null;
        }

        /* @var $audit Audit */
        $audit = Yii::createObject([
            'class' => Audit::class, 'user_id' => $user_id,
            'category' => $category,
            'message' => $message,
            'dataUnencoded' => $this->redactData($data),]);

        $audit->ip = Yii::$app instanceof ConsoleApplication
            ? 'Console request'
            : Yii::$app->request->remoteIP;


        if (!$audit->save()) {
            throw new InvalidArgumentException(Yii::t('audit', 'Cannot log audit entry: {reason}', [
                'reason' => json_encode($audit->errors, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
            ]));
        }
    }

    /**
     * Add an audit entry including POST data. Data is automatically redacted according to `$this->redactedValues`.
     *
     * @param string $category
     * @param string $message
     * @param array $data
     * @param int $user_id
     * @throws InvalidConfigException
     */
    public function logPost($category, $message = null, $data = null, $user_id = null)
    {
        if (!$data) {
            $data = [];
        }
        $data['POST'] = Yii::$app->request->post();

        $this->log($category, $message, $data, $user_id);
    }

    /**
     * @param array $data
     * @return array
     */
    protected function redactData($data)
    {
        foreach ($this->redactedValues as $path => $value) {
            if (is_integer($path)) {
                $path = $value;
                $value = '<value redacted>';
            }
            if (ArrayHelper::getValue($data, $path)) {
                ArrayHelper::setValue($data, $path, $value);
            }
        }

        return $data;
    }

    /**
     * @param WebApplication|ConsoleApplication $app
     * @throws InvalidConfigException
     */
    protected function injectTranslations($app)
    {
        if (!isset($app->get('i18n')->translations['audit*'])) {
            $app->get('i18n')->translations['audit*'] = [
                'class' => PhpMessageSource::class,
                'basePath' => __DIR__ . '/messages',
                'sourceLanguage' => 'en'
            ];
        }
    }

    /**
     * @param WebApplication $app
     * @throws InvalidConfigException
     */
    protected function addUrlRules($app)
    {
        $urlRuleConfig = [
            'class' => GroupUrlRule::class,
            'prefix' => $this->urlPrefix,
            'rules' => $this->urlRules,
        ];

        if ($this->urlPrefix !== $this->id) {
            $urlRuleConfig['routePrefix'] = $this->id;
        }

        $rule = Yii::createObject($urlRuleConfig);

        $app->urlManager->addRules([$rule], false);
    }

    /**
     * Returns the config of a command or false, if no such command is configured.
     *
     * @param ConsoleApplication $app
     * @param string $command
     * @return array|boolean
     */
    protected function retrieveCommandConfig($app, $command)
    {
        if (!empty($app->controllerMap[$command])) {
            return $app->controllerMap[$command];
        }
        $coreCommands = $app->coreCommands();
        if (isset($coreCommands[$command])) {
            return $coreCommands[$command];
        }

        return false;
    }
}
