<?php

namespace wdigital\cms\audit;

use wdigital\cms\audit\models\Audit;
use Yii;
use yii\base\BootstrapInterface;
use yii\base\InvalidParamException;
use yii\base\Module as BaseModule;
use yii\console\Application as ConsoleApplication;
use yii\helpers\ArrayHelper;
use yii\i18n\PhpMessageSource;
use yii\web\Application as WebApplication;
use yii\web\GroupUrlRule;

class Module extends BaseModule implements BootstrapInterface
{
    /**
     * Values to be redacted out of the audit data (not stored).
     * @var array
     */
    public $redactedValues = [
        'POST.password' => '<password redacted>',
    ];

    /**
     * The prefix for the module URLs.
     * @see [[GroupUrlRule::prefix]]
     * @var string
     */
    public $urlPrefix = 'audit';

    /**
     * The rules to be used in URL management.
     * @var array
     */
    public $urlRules = [
        '' => 'default/index',
        '<action:\w+>' => 'default/<action>',
    ];

    /**
     * The namespace of migrations for this module.
     * @var string
     */
    public $migrationNamespace = __NAMESPACE__ . '\\migrations';

    /**
     * Console application command (controller ID) for running migrations.
     * @var string
     */
    public $migrateCommand = 'migrate';

    /**
     * @inheritdoc
     * @param WebApplication|ConsoleApplication $app
     */
    public function bootstrap($app)
    {
        if ($app instanceof WebApplication) {
            $this->addUrlRules($app);
        } elseif ($app instanceof ConsoleApplication) {
            $this->injectMigrations($app);
        }
    }

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();

        $app = Yii::$app;
        if ($app instanceof WebApplication) {
            $this->injectTranslations($app);
        }
    }

    /**
     * Add an audit entry. Data is automatically redacted according to `$this->redactedValues`.
     *
     * @param string $category
     * @param string $message
     * @param array $data
     * @param int $user_id
     */
    public function log($category, $message = null, $data = null, $user_id = null)
    {
        if (!$user_id) {
            $user_id = Yii::$app->user->id;
        }

        /* @var $audit Audit */
        $audit = \Yii::createObject([
            'class' => Audit::class,
            'user_id' => $user_id,
            'ip' => Yii::$app->request->remoteIP,
            'category' => $category,
            'message' => $message,
            'dataUnencoded' => $this->redactData($data),
        ]);

        if (!$audit->save()) {
            throw new InvalidParamException(Yii::t('audit', 'Cannot log audit entry: {reason}', [
                'reason' => json_encode($audit->errors, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
            ]));
        }
    }

    /**
     * Add an audit entry including POST data. Data is automatically redacted according to `$this->redactedValues`.
     *
     * @param string $category
     * @param string $message
     * @param array $data
     * @param int $user_id
     */
    public function logPost($category, $message = null, $data = null, $user_id = null)
    {
        if (!$data) {
            $data = [];
        }
        $data['POST'] = Yii::$app->request->get();

        $this->log($category, $message, $data, $user_id);
    }

    protected function redactData($data)
    {
        foreach ($this->redactedValues as $path => $value) {
            if (is_integer($path)) {
                $path = $value;
                $value = '<value redacted>';
            }
            if (ArrayHelper::getValue($data, $path)) {
                ArrayHelper::setValue($data, $path, $value);
            }
        }

        return $data;
    }

    protected function injectTranslations($app)
    {
        if (!isset($app->get('i18n')->translations['audit*'])) {
            $app->get('i18n')->translations['audit*'] = [
                'class' => PhpMessageSource::class,
                'basePath' => __DIR__ . '/messages',
                'sourceLanguage' => 'en'
            ];
        }
    }

    protected function addUrlRules($app)
    {
        $urlRuleConfig = [
            'class' => GroupUrlRule::class,
            'prefix' => $this->urlPrefix,
            'rules' => $this->urlRules,
        ];

        if ($this->urlPrefix !== $this->id) {
            $urlRuleConfig['routePrefix'] = $this->id;
        }

        $rule = Yii::createObject($urlRuleConfig);

        $app->urlManager->addRules([$rule], false);
    }

    /**
     * Called when bootstrapping a console application with this module.
     *
     * @param type $app
     */
    protected function injectMigrations($app)
    {
        $migrateCommand = $this->migrateCommand;

        $migrateConfig = $this->retrieveCommandConfig($app, $migrateCommand);
        if ($migrateConfig) {
            $migrateConfig = is_string($migrateConfig)
                ? ['class' => $migrateConfig]
                : (array)$migrateConfig; // better safe than sorry

            $migrationNamespaces = ArrayHelper::getValue($migrateConfig, 'migrationNamespaces', []);
            $migrationNamespaces[] = $this->migrationNamespace;
            $migrateConfig['migrationNamespaces'] = $migrationNamespaces;
            $app->controllerMap[$migrateCommand] = $migrateConfig;
        }
    }

    /**
     * Returns the config of a command or false, if no such command is configured.
     *
     * @param ConsoleApplication $app
     * @param string $command
     * @return array|boolean
     */
    protected function retrieveCommandConfig($app, $command)
    {
        if (!empty($app->controllerMap[$command])) {
            return $app->controllerMap[$command];
        } else {
            $coreCommands = $app->coreCommands();
            if (isset($coreCommands[$command])) {
                return $coreCommands[$command];
            }
        }

        return false;
    }
}
